% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nas.R
\name{is_na}
\alias{is_na}
\alias{num_na}
\alias{which_na}
\alias{which_not_na}
\alias{any_na}
\alias{all_na}
\alias{row_na_counts}
\alias{col_na_counts}
\alias{row_all_na}
\alias{col_all_na}
\alias{row_any_na}
\alias{col_any_na}
\title{Efficient functions for dealing with missing values.}
\usage{
is_na(x)

num_na(x)

which_na(x)

which_not_na(x)

any_na(x)

all_na(x)

row_na_counts(x)

col_na_counts(x)

row_all_na(x)

col_all_na(x)

row_any_na(x)

col_any_na(x)
}
\arguments{
\item{x}{A vector, matrix or data frame.}
}
\value{
Number or location of \code{NA} values.
}
\description{
\code{is_na()} is a parallelised alternative to \code{is.na()}. \cr
\code{num_na(x)} is a faster and more efficient \code{sum(is.na(x))}. \cr
\code{which_na(x)} is a more efficient \code{which(is.na(x))} \cr
\code{which_not_na(x)} is a more efficient \code{which(!is.na(x))} \cr
\code{row_na_counts(x)} is a more efficient \code{rowSums(is.na(x))} \cr
\code{row_all_na()} returns a logical vector indicating which rows are empty
and have only \code{NA} values. \cr
\code{row_any_na()} returns a logical vector indicating which rows have at least
1 \code{NA} value. \cr
The \code{col_} variants are the same, but operate by-column.
}
\details{
These functions are designed primarily for programmers, to increase the speed
and memory-efficiency of \code{NA} handling. \cr
Most of these functions can be parallelised through \code{options(cheapr.cores)}. \cr
When \code{x} is a list, \code{num_na}, \code{any_na} and \code{all_na} will recurse through a
potentially nested list for \code{NA} values. \cr

For the functions that operate on data frames like \code{row_na_counts},
if \code{x} contains a list-variable, an element of that list is considered to be
\code{NA} when all nested elements of that list-element contain \code{NA} values.
\subsection{Common use-cases}{

To replicate \code{complete.cases(x)}, use \code{!row_any_na(x)}. \cr
To find rows with any empty values,
use \code{which_(row_any_na(df))}. \cr
To find empty rows use \code{which_(row_all_na(df))}.
}

\subsection{\code{is_na}}{

\code{is_na} Is an S3 generic function. It will internally fall back on
using \code{is.na} if it can't find a suitable method.
Alternatively you can write your own \code{is_na} method.
For example there is a method for \code{vctrs_rcrd}
objects that simply converts it to a data frame and then calls \code{row_all_na()}.
There is also a \code{POSIXlt} method for \code{is_na} that is much faster than \code{is.na}.
}
}
\examples{
library(cheapr)
library(bench)

x <- 1:10
x[c(1, 5, 10)] <- NA
num_na(x)
which_na(x)
which_not_na(x)

row_nas <- row_na_counts(airquality)
col_nas <- col_na_counts(airquality)
names(row_nas) <- rownames(airquality)
names(col_nas) <- colnames(airquality)
row_nas
col_nas

df <- airquality[, 1:2]

# Number of NAs in data
num_na(df)
# Which rows are empty?
row_na <- row_all_na(df)
df[which_(row_na), ]

# Removing the empty rows
df[which_(row_na, invert = TRUE), ]

}
