\name{maxCorProj}
\alias{maxCorProj}
\title{(Robust) maximum correlation via projections through the data points}
\usage{
  maxCorProj(x, y,
    method = c("spearman", "kendall", "quadrant", "M", "pearson"),
    control = list(...), useL1Median = TRUE,
    fallback = FALSE, ...)
}
\arguments{
  \item{x,y}{each can be a numeric vector, matrix or data
  frame.}

  \item{method}{a character string specifying the
  correlation functional to maximize.  Possible values are
  \code{"spearman"} for the Spearman correlation,
  \code{"kendall"} for the Kendall correlation,
  \code{"quadrant"} for the quadrant correlation,
  \code{"M"} for the correlation based on a bivariate
  M-estimator of location and scatter with a Huber loss
  function, or \code{"pearson"} for the classical Pearson
  correlation (see \code{\link{corFunctions}}).}

  \item{control}{a list of additional arguments to be
  passed to the specified correlation functional.  If
  supplied, this takes precedence over additional arguments
  supplied via the \code{\dots} argument.}

  \item{useL1Median}{a logical indicating whether the
  \eqn{L_{1}}{L1} medians should be used as the centers of
  the data sets (defaults to \code{TRUE}).  If
  \code{FALSE}, the columnwise centers are used instead
  (columnwise means if \code{method} is \code{"pearson"}
  and columnwise medians otherwise).}

  \item{fallback}{logical; if a correlation functional
  other than the Pearson correlation is maximized, the data
  are first robustly standardized via median and MAD.  This
  indicates whether standardization via mean and standard
  deviation should be performed as a fallback mode for
  variables whose MAD is zero (e.g., for dummy variables).
  Note that if the Pearson correlation is maximized, the
  data are always standardized via mean and standard
  deviation.}

  \item{\dots}{additional arguments to be passed to the
  specified correlation functional.}
}
\value{
  An object of class \code{"maxCor"} with the following
  components:

  \item{cor}{a numeric giving the maximum correlation
  estimate.}

  \item{a}{numeric; the weighting vector for \code{x}.}

  \item{b}{numeric; the weighting vector for \code{y}.}

  \item{call}{the matched function call.}
}
\description{
  Compute the maximum correlation between two data sets via
  projection pursuit based on projections through the data
  points, with a focus on robust and nonparametric methods.
}
\details{
  First the candidate projection directions are defined for
  each data set from the respective center through each
  data point.  Then the algorithm scans all \eqn{n^2}
  possible combinations for the maximum correlation, where
  \eqn{n} is the number of observations.
}
\examples{
## generate data
library("mvtnorm")
set.seed(1234)  # for reproducibility
p <- 3
q <- 2
m <- p + q
sigma <- 0.5^t(sapply(1:m, function(i, j) abs(i-j), 1:m))
xy <- rmvnorm(100, sigma=sigma)
x <- xy[, 1:p]
y <- xy[, (p+1):m]

## Spearman correlation
maxCorProj(x, y, method = "spearman")
maxCorProj(x, y, method = "spearman", consistent = TRUE)

## Pearson correlation
maxCorProj(x, y, method = "pearson")
}
\author{
  Andreas Alfons
}
\seealso{
  \code{\link{maxCorGrid}}, \code{\link{ccaProj}},
  \code{\link{corFunctions}},
}
\keyword{multivariate}
\keyword{robust}

