#' @title Compute Data for a Grid Layer
#' @description This function computes data to match a grid layer (as outputed 
#' by \link{getGridLayer} )
#' according to the surface intersections.
#' @name getGridData
#' @param x a list generated by the \link{getGridLayer} function.
#' @param df a data frame that contains the values to adapt to the grid. It must 
#' correspond to the spdf argument in \link{getGridLayer}.
#' @param dfid identifier field in df, default to the first column 
#' of df. (optional)
#' @param var name of the numeric field in df to adapt to the grid.
#' @return A data frame is returned. id_cell are ids of the grid, the two other 
#' variable are the share of the variable in each cell and the share of the 
#' variable in each cell divided by its area (in map units).
#' @seealso \link{getGridLayer}
#' @examples
#' \dontrun{
#' data(nuts2006)
#' # Create a grid layer
#' mygrid <- getGridLayer(spdf=nuts2.spdf,cellsize = 200000)
#' # Compute data for the grid layer
#' datagrid.df <- getGridData(mygrid, nuts2.df, "pop2008",dfid=NULL)
#' 
#' # Plot total population
#' plot(mygrid$spdf, col="#CCCCCC",border="white")
#' propSymbolsLayer(spdf = mygrid$spdf, df = datagrid.df, legend.style = "e",
#'                  legend.pos = "right", border = "white",legend.title.txt = "Total population",
#'                  var = "pop2008", inches=0.01, col="black", add=TRUE)
#' 
#' # Plot dentsity of population 
#' ## conversion from square meter to square kilometers
#' datagrid.df$densitykm <- datagrid.df$pop2008_density*1000*1000 
#' cols <- carto.pal(pal1 = "taupe.pal", n1 = 6)
#' choroLayer(spdf = mygrid$spdf,df = datagrid.df,var = "densitykm", add=FALSE,
#'            border = "grey80",col=cols,
#'            legend.pos = "right", method = "q6",
#'            legend.title.txt = "Population density")
#'            }
#' @export
getGridData <- function(x, df, dfid=NULL, var) {
  if (is.null(dfid)){dfid <- names(df)[1]}
  tab <- x$df
  tab <- data.frame(tab, var=df[match(tab$id_geo, df[,dfid]),var])
  tab$var_parts <- (tab$var*tab$area_pct/100)
  
  # Construction du jeu de données : aggregate by id_cell
  grid_data<-aggregate(tab$var_parts, by = list(key = tab$id_cell), sum, 
                       simplify=TRUE)
  colnames(grid_data) <- c("id_cell","var")
  
  # Ajoutr d'une colonne de densité si on veut représenter en applat
  grid_data <- data.frame(grid_data, 
                          cell_area=
                            x$spdf@data[match(
                              grid_data$id_cell, x$spdf@data$id),
                              "cell_area"])
  grid_data$var_density <- (grid_data$var/grid_data$cell_area)
  grid_data <- grid_data[,c("id_cell","var","var_density")]
  colnames(grid_data) <- c("id_cell",var,paste(var,"density",sep="_"))
  return(grid_data)
}

