% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/proportion_ci.R
\name{proportion_ci}
\alias{proportion_ci}
\alias{proportion_ci_wald}
\alias{proportion_ci_wilson}
\alias{proportion_ci_clopper_pearson}
\alias{proportion_ci_agresti_coull}
\alias{proportion_ci_jeffreys}
\alias{proportion_ci_strat_wilson}
\alias{is_binary}
\title{Functions for Calculating Proportion Confidence Intervals}
\usage{
proportion_ci_wald(x, conf.level = 0.95, correct = FALSE)

proportion_ci_wilson(x, conf.level = 0.95, correct = FALSE)

proportion_ci_clopper_pearson(x, conf.level = 0.95)

proportion_ci_agresti_coull(x, conf.level = 0.95)

proportion_ci_jeffreys(x, conf.level = 0.95)

proportion_ci_strat_wilson(
  x,
  strata,
  weights = NULL,
  conf.level = 0.95,
  max.iterations = 10L,
  correct = FALSE
)

is_binary(x)
}
\arguments{
\item{x}{vector of a binary values, i.e. a logical vector, or numeric with values \code{c(0, 1)}}

\item{conf.level}{(\code{numeric})\cr
a scalar in \verb{(0, 1)} indicating the confidence level.
Default is \code{0.95}}

\item{correct}{(\code{flag})\cr include the continuity correction. For further information, see for example
\code{\link[stats:prop.test]{stats::prop.test()}}.}

\item{strata}{(\code{factor})\cr variable with one level per stratum and same length as \code{x}.}

\item{weights}{(\code{numeric} or \code{NULL})\cr weights for each level of the strata. If \code{NULL}, they are
estimated using the iterative algorithm that
minimizes the weighted squared length of the confidence interval.}

\item{max.iterations}{(\code{count})\cr maximum number of iterations for the iterative procedure used
to find estimates of optimal weights.}
}
\value{
Confidence interval of a proportion.
}
\description{
Functions to calculate different proportion confidence intervals for use in \code{ard_proportion()}.
}
\section{Functions}{
\itemize{
\item \code{proportion_ci_wald()}: Calculates the Wald interval by following the usual textbook definition
for a single proportion confidence interval using the normal approximation.

\deqn{\hat{p} \pm z_{\alpha/2} \sqrt{\frac{\hat{p}(1 - \hat{p})}{n}}}

\item \code{proportion_ci_wilson()}: Calculates the Wilson interval by calling \code{\link[stats:prop.test]{stats::prop.test()}}.
Also referred to as Wilson score interval.

\deqn{\frac{\hat{p} +
\frac{z^2_{\alpha/2}}{2n} \pm z_{\alpha/2} \sqrt{\frac{\hat{p}(1 - \hat{p})}{n} +
\frac{z^2_{\alpha/2}}{4n^2}}}{1 + \frac{z^2_{\alpha/2}}{n}}}

\item \code{proportion_ci_clopper_pearson()}: Calculates the Clopper-Pearson interval by calling \code{\link[stats:binom.test]{stats::binom.test()}}.
Also referred to as the \code{exact} method.

\deqn{
\left( \frac{k}{n} \pm z_{\alpha/2} \sqrt{\frac{\frac{k}{n}(1-\frac{k}{n})}{n} +
\frac{z^2_{\alpha/2}}{4n^2}} \right)
/ \left( 1 + \frac{z^2_{\alpha/2}}{n} \right)}

\item \code{proportion_ci_agresti_coull()}: Calculates the \code{Agresti-Coull} interval (created by \verb{Alan Agresti} and \verb{Brent Coull}) by
(for 95\% CI) adding two successes and two failures to the data and then using the Wald formula to construct a CI.

\deqn{
\left( \frac{\tilde{p} + z^2_{\alpha/2}/2}{n + z^2_{\alpha/2}} \pm
z_{\alpha/2} \sqrt{\frac{\tilde{p}(1 - \tilde{p})}{n} +
\frac{z^2_{\alpha/2}}{4n^2}} \right)}

\item \code{proportion_ci_jeffreys()}: Calculates the Jeffreys interval, an equal-tailed interval based on the
non-informative Jeffreys prior for a binomial proportion.

\deqn{\left( \text{Beta}\left(\frac{k}{2} + \frac{1}{2}, \frac{n - k}{2} + \frac{1}{2}\right)_\alpha,
\text{Beta}\left(\frac{k}{2} + \frac{1}{2}, \frac{n - k}{2} + \frac{1}{2}\right)_{1-\alpha} \right)}

\item \code{proportion_ci_strat_wilson()}: Calculates the stratified Wilson confidence
interval for unequal proportions as described in
Xin YA, Su XG. Stratified Wilson and Newcombe confidence intervals
for multiple binomial proportions. \emph{Statistics in Biopharmaceutical Research}. 2010;2(3).

\deqn{\frac{\hat{p}_j + \frac{z^2_{\alpha/2}}{2n_j} \pm
z_{\alpha/2} \sqrt{\frac{\hat{p}_j(1 - \hat{p}_j)}{n_j} +
\frac{z^2_{\alpha/2}}{4n_j^2}}}{1 + \frac{z^2_{\alpha/2}}{n_j}}}

\item \code{is_binary()}: Helper to determine if vector is binary (logical or 0/1)

}}
\examples{
\dontshow{if (do.call(asNamespace("cardx")$is_pkg_installed, list(pkg = "broom"))) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
x <- c(
  TRUE, TRUE, TRUE, TRUE, TRUE,
  FALSE, FALSE, FALSE, FALSE, FALSE
)

proportion_ci_wald(x, conf.level = 0.9)
proportion_ci_wilson(x, correct = TRUE)
proportion_ci_clopper_pearson(x)
proportion_ci_agresti_coull(x)
proportion_ci_jeffreys(x)
\dontshow{\}) # examplesIf}
# Stratified Wilson confidence interval with unequal probabilities

set.seed(1)
rsp <- sample(c(TRUE, FALSE), 100, TRUE)
strata_data <- data.frame(
  "f1" = sample(c("a", "b"), 100, TRUE),
  "f2" = sample(c("x", "y", "z"), 100, TRUE),
  stringsAsFactors = TRUE
)
strata <- interaction(strata_data)
n_strata <- ncol(table(rsp, strata)) # Number of strata

proportion_ci_strat_wilson(
  x = rsp, strata = strata,
  conf.level = 0.90
)

# Not automatic setting of weights
proportion_ci_strat_wilson(
  x = rsp, strata = strata,
  weights = rep(1 / n_strata, n_strata),
  conf.level = 0.90
)

}
