% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/adult_weight.R
\name{adult_weight}
\alias{adult_weight}
\title{Dynamic Adult Weight Change Model}
\usage{
adult_weight(bw, ht, age, sex, EIchange = matrix(0, ncol =
  abs(ceiling(days/dt)), nrow = length(bw)), NAchange = matrix(0, ncol =
  abs(ceiling(days/dt)), nrow = length(bw)), EI = NA, fat = rep(NA,
  length(bw)), PAL = rep(1.5, length(bw)), pcarb_base = rep(0.5,
  length(bw)), pcarb = pcarb_base, days = 365, dt = 1,
  checkValues = TRUE)
}
\arguments{
\item{bw}{(vector) Body weight for model (kg)}

\item{ht}{(vector) Height for model (m)}

\item{age}{(vector) Age of individual (yrs)}

\item{sex}{(vector) Sex either \code{"female"} or \code{"male"}}

\item{EIchange}{(matrix) Matrix of caloric intake change (kcals)}

\item{NAchange}{(matrix) Vector of sodium intake change (mg)

\strong{ Optional }}

\item{EI}{(vector) Energy Intake at Baseline.}

\item{fat}{(vector) Vector containing fat mass. Recall that}

\item{PAL}{(vector) Physical activity level.}

\item{pcarb_base}{(vector) Percent carbohydrates at baseline.}

\item{pcarb}{(vector) Percent carbohydrates after intake change.}

\item{days}{(double) Days to run the model.}

\item{dt}{(double) Time step for model; default 1 day (\code{dt = 1})}

\item{checkValues}{(boolean) Check whether the values from the model are biologically feasible.}
}
\description{
Estimates weight change given energy and sodium intake changes at 
individual level.
}
\details{
\code{EIchange} and \code{NAchange} must be consumption
change matrices. Each row should represent consumption at each
day. That is, each row of \code{EIchange} and \code{NAchange} 
represents a day in consumption change since baseline. Consumption
change is non-cummulative and it's all from baseline. 
As an example, \code{EIchange <- rep(-100, 50)} represents that 
each day \code{-100} kcals are reduced from consumption.
}
\examples{
#EXAMPLE 1: INDIVIDUAL MODELLING
#--------------------------------------------------------
#For one female in a diet of 100 kcal reduction. 
adult_weight(80, 1.8, 40, "female", rep(-100, 365))

#Same female also reducing sodium in -25mg
adult_weight(80, 1.8, 40, "female", rep(-100, 365), rep(-25, 365))

#Same female modelled for 400 days
adult_weight(80, 1.8, 40, "female", rep(-100, 400), rep(-25, 400), days = 400)

#Same female reducing -50 kcals per 100 days and not reducing sodium
kcalvec <-c(rep(-50, 100), rep(-100, 100), rep(-150, 100), rep(-200, 100))
adult_weight(80, 1.8, 40, "female", kcalvec, days = 400)

#Same female with known energy intake
adult_weight(80, 1.8, 40, "female", rep(-100, 365), rep(-25, 365), EI = 2000)

#Same female with known fat mass
adult_weight(80, 1.8, 40, "female", rep(-100, 365), rep(-25, 365), fat = 32)

#Same female with known fat mass and known energy consumption
adult_weight(80, 1.8, 40, "female", rep(-100, 365), rep(-25, 365), EI = 2000, fat = 32)

#EXAMPLE 2: DATASET MODELLING
#--------------------------------------------------------

#Antropometric data
weights <- c(45, 67, 58, 92, 81)
heights <- c(1.30, 1.73, 1.77, 1.92, 1.73)
ages    <- c(45, 23, 66, 44, 23)
sexes   <- c("male", "female", "female", "male", "male") 

#Matrix of energy consumption reduction: 
EIchange <- rbind(rep(-100, 365), rep(-200, 365), rep(-200, 365), 
                  rep(-123, 365), rep(-50, 365))

#Returns a weight change matrix and other matrices
model_weight <- adult_weight(weights, heights, ages, sexes, 
                             EIchange)["Body_Weight"][[1]]

}
\references{
Chow, Carson C, and Kevin D Hall. 2008. \emph{The Dynamics of Human Body Weight Change.} PLoS Comput Biol 4 (3):e1000045.

Hall, Kevin D. 2010. \emph{Predicting Metabolic Adaptation, Body Weight Change, and Energy Intake in Humans.}
   American Journal of Physiology-Endocrinology and Metabolism 298 (3). Am Physiological Soc: E449–E466.

Hall, Kevin D, and Peter N Jordan. 2008. \emph{Modeling Weight-Loss Maintenance to Help Prevent Body Weight Regain.}
    The American Journal of Clinical Nutrition 88 (6). Am Soc Nutrition: 1495–1503.
    
Hall, Kevin D, Gary Sacks, Dhruva Chandramohan, Carson C Chow, Y Claire Wang, Steven L Gortmaker, and Boyd A Swinburn. 2011.
\emph{Quantification of the Effect of Energy Imbalance on Bodyweight.} The Lancet 378 (9793). Elsevier: 826–37.
\emph{A New Predictive Equation for Resting Energy Expenditure in Healthy Individuals.} 
The American Journal of Clinical Nutrition 51 (2). Am Soc Nutrition: 241–47.
}
\seealso{
\code{\link{model_plot}} for plotting the results and 
\code{\link{model_mean}} for aggregate data estimation. \code{\link{child_weight}} 
implements a similar model for children.
}
\author{
Dalia Camacho-García-Formentí \email{daliaf172@gmail.com}

Rodrigo Zepeda-Tello \email{rzepeda17@gmail.com}
}
