% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/design.r
\name{design}
\alias{design}
\title{General block and treatment designs.}
\usage{
design(
  treatments,
  blocks = NULL,
  treatments_model = NULL,
  weighting = 0.5,
  searches = NULL,
  seed = NULL,
  jumps = 1
)
}
\arguments{
\item{treatments}{a single treatment factor or data frame for the candidate set 
of treatment factor combinations assuming any combination of qualitative or quantitative factor levels.}

\item{blocks}{a single block factor or data frame for the required combinations of 
block factors in the required order of fitting assuming quantitative block factor levels only.}

\item{treatments_model}{a character vector containing one or more nested treatment model formula.}

\item{weighting}{a weighting factor between 0 and 1 for weighting the 2-factor interaction effects of
factorial blocks.}

\item{searches}{the maximum number of local optima searched at each stage of an
optimization.}

\item{seed}{an integer initializing the random number generator.}

\item{jumps}{the number of pairwise random treatment swaps used to escape a local maxima.}
}
\value{
\item{Treatments}{The treatments included in the design and the replication of each individual 
 treatment taken in de-randomized standard order.}
\item{Design}{The design layout showing the randomized allocation of treatments to blocks and plots.}
\item{Treatments_model}{The fitted treatment model, the number of model parameters (DF)
  and the D-efficiency of each sequentially fitted treatment model}
\item{Blocks_model}{The blocks sub-model design and 
 the D- and A-efficiency factors of each successively fitted sub-blocks model.}
\item{seed}{Numerical seed for random number generator.}
\item{searches}{Maximum number of searches in each stratum.}
\item{jumps}{Number of random treatment swaps to escape a local maxima.}
}
\description{
Constructs general D-optimal designs for any feasible linear treatment model with any feasible combinations of block 
factors.
}
\details{
\code{treatments} is a factor or data frame containing one or more qualitative
or quantitative level treatment factors defining the candidate treatment set. The 
required treatment design is selected from the candidate treatment set without replacement.

 i) If the candidate set is the same size as the required design, the full candidate set will be
selected. 

ii) If the candidate set is larger than the required design, a subset of the candidate set 
will be selected by optimizing the treatment D-optimality criterion. 

iii) If the candidate set is smaller than the required design size, the candidate set will be replicated 
until the candidate set equals or exceeds the required design size.

\code{blocks} is a factor or data frame containing one or more qualitative level block factors, added
sequentially. If the blocks are fully nested or if the design has additive crossed blocks,
each added block factor is optimized by swaps made within the levels of all previously added blocks. If,
however, a design has crossed blocks with estimable interaction effects, the relative importance of the main
factorial block effects versus the 2-factor block interaction effects is weighted by a parameter w in the range 0 to 1. 

i) If w = 0, the design is a simple additive block effects. 

ii) If w = 1, the design is a simple interaction blocks design for the crossed blocks interaction effects

iii) If 0 < w < 1, the 2-factor block interaction effects are weighted relative to the additive block effects with
the importance of the interaction effects assumed to increase as w increases from 0 to 1. 

The length of the \code{blocks} object defines the total number of plots in the design. 
 
\code{treatments_model} is a character vector containing one or more nested treatments formula where
each model formula, taken in order, is optimized with the treatment factors of all previously
optimized model formula remaining constant. Sequential model fitting provides improved flexibility
for fitting factors or variables of different status or importance (see examples below).

The treatment design criterion for each treatment model is the generalized variance of the information matrix for that design 
and the design efficiency is the ratio of the generalized variance of the full candidate treatment model
relative to the generalized variance of the optimized design. The efficiency will be less than or equal to 1 for factorial
models but may exceed 1 for polynomial models. 

For more details see \code{vignette(package = "blocksdesign")}
}
\examples{
## For optimum results, the number of searches may need to be increased.

## 4 replicates of 12 treatments with 16 nested blocks of size 3
## rectangular lattice see Plan 10.10 Cochran and Cox 1957.
blocks = data.frame(Main = gl(4,12), Sub = gl(16,3))
design(treatments = gl(12,1,48), blocks)

## 3 replicates of 15 treatments in 3 main blocks with 3 nested blocks and one control treatment
blocks=data.frame( Main = gl(3,18,54),Sub = gl(9,6,54))
treatments=factor(rep(c(1:15,rep("control",3)),3),levels = c(1:15,"control") )
Z=design(treatments,blocks)
incid=table(interaction(Z$Design$Main,Z$Design$Sub,lex.order = TRUE),Z$Design$treatments)
Z # print design
incid # print incidences of treatments in blocks
crossprod(incid) # print pairwise concurrences within blocks
 
## 4 x 12 design for 4 replicates of 12 treatments with 3 plots in each intersection block
## The optimal design is Trojan with known A-efficiency = 22/31 for the intersection blocks
blocks = data.frame(Rows = gl(4,12), Cols = gl(4,3,48))
Z=design(treatments =gl(12,1,48),blocks)
incid=table(interaction(Z$Design$Rows,Z$Design$Cols,lex.order = TRUE),Z$Design$treatments)
Z # print design
incid # print incidences of treatments in blocks
crossprod(incid) # print pairwise concurrences within blocks
## as above but showing 3 sub-columns nested within each main column
blocks = data.frame(Rows = gl(4,12), Cols = gl(4,3,48), subCols = gl(12,1,48))
\donttest{Z=design(treatments = gl(12,1,48),blocks,searches=200)
Z # print design}

## 4 x 13 Row-and-column design for 4 replicates of 13 treatments 
## Youden design Plan 13.5 Cochran and Cox (1957).
blocks = data.frame(Rows = gl(4,13), Cols = gl(13,1,52))
\donttest{Z=design(treatments = gl(13,1,52),blocks,searches = 700)
incid=table(Z$Design$Cols,Z$Design$treatments)
Z # print design
crossprod(incid) # print pairwise concurrences of treatments within column blocks (BIB's)
tcrossprod(incid) # print pairwise concurrences of column blocks within treatments (Dual design)}

## 48 treatments in 2 replicate blocks with 2 nested rows in each replicate and 3 main columns
##  (Reps/Rows) x Cols
blocks = data.frame(Reps = gl(2,48), Rows = gl(4,24,96), Cols = gl(3,8,96))
design(treatments=gl(48,1,96),blocks,searches=5)

## 48 treatments in 2 replicate blocks with 2 main columns
## The default weighting gives non-estimable Reps:Cols effects due to inherent aliasing
## Increased weighting gives estimable Reps:Cols effects but non-orthogonal main effects
blocks = data.frame(Reps = gl(2,48), Cols = gl(2,24,96))
design(treatments=gl(48,1,96),blocks,searches=5)
design(treatments=gl(48,1,96),blocks,searches=5,weighting=.9)

## Factorial treatment designs defined by a single factorial treatment model

## Main effects of five 2-level factors in a half-fraction in 2/2/2 nested blocks design 
## (may require 100's of repeats to find a fully orthogonal solution - a VERY long wait!)
treatments = expand.grid(F1 = gl(2,1), F2 = gl(2,1),F3 = gl(2,1), F4 = gl(2,1), F5 = gl(2,1))
blocks = data.frame(b1 = gl(2,8),b2 = gl(4,4),b3 = gl(8,2))
model=" ~ F1 + F2 + F3 + F4 + F5"
\donttest{repeat {z = design(treatments,blocks,treatments_model=model,searches=50)
if ( isTRUE(all.equal(z$Blocks_model[3,3],1) ) ) break }
print(z)}
 
# Second-order model for five qualitative 2-level factors in 4 randomized blocks
treatments = expand.grid(F1 = gl(2,1), F2 = gl(2,1),F3 = gl(2,1), F4 = gl(2,1), F5 = gl(2,1))
blocks = data.frame(blocks = gl(4,8))
model = " ~ (F1 + F2 + F3 + F4 + F5)^2"
design(treatments,blocks,treatments_model=model,searches = 10)

# Main effects of five 2-level factors in a half-fraction of a 4 x 4 row-and column design.
treatments = expand.grid(F1 = gl(2,1), F2 = gl(2,1),F3 = gl(2,1), F4 = gl(2,1), F5 = gl(2,1))
blocks = data.frame(rows = gl(4,4), cols = gl(4,1,16))
model = "~ F1 + F2 + F3 + F4 + F5"
\donttest{repeat {z = design(treatments,blocks,treatments_model=model,searches=50)
if ( isTRUE(all.equal(z$Blocks_model[2,3],1) ) ) break }
 print(z)}

# Quadratic regression for three 3-level numeric factor assuming a 10/27 fraction
treatments = expand.grid(A = 1:3, B = 1:3, C = 1:3)
blocks=data.frame(main=gl(1,10))
model = " ~ ( A + B + C)^2 + I(A^2) + I(B^2) + I(C^2)"
design(treatments,blocks,treatments_model=model,searches=10) 

# Quadratic regression for three 3-level numeric factor crossed with a qualitative 2-level factor
treatments = expand.grid(F = factor(1:2), A = 1:3, B = 1:3, C = 1:3)
blocks=data.frame(main=gl(1,18))
model = " ~ F + A + B + C + F:A + F:B + F:C + A:B + A:C + B:C + I(A^2) + I(B^2) + I(C^2)"
design(treatments,blocks,treatments_model=model,searches=5) 

# 1st-order model for 1/3rd fraction of four qualitative 3-level factors in 3  blocks
treatments = expand.grid(F1 = gl(3,1), F2 = gl(3,1), F3 = gl(3,1), F4 = gl(3,1))
blocks = data.frame(main = gl(3,9))
model = " ~ F1 + F2 + F3 + F4"
\donttest{ design(treatments,blocks,treatments_model=model,searches=25)}

# 2nd-order model for a 1/3rd fraction of five qualitative 3-level factors in 3 blocks
# (may require many repeats to find a fully orthogonal solution)
treatments = expand.grid(F1 = gl(3,1), F2 = gl(3,1),F3 = gl(3,1), F4 = gl(3,1), F5 = gl(3,1))
blocks=data.frame(main=gl(3,27))
model = " ~ (F1 + F2 + F3 + F4 + F5)^2"
\donttest{ repeat {z = design(treatments,blocks,treatments_model=model,searches=50)
if ( isTRUE(all.equal(z$Blocks_model[1,3],1) ) ) break}
 print(z) }

# 2nd-order model for two qualitative and two quantitative level factors in 2 blocks of size 18
treatments = expand.grid(F1 = factor(1:2), F2 = factor(1:3), V1 = 1:3, V2 = 1:4)
blocks = data.frame(main = gl(2,18))
model = " ~ (F1 + F2 + V1 + V2)^2 +  I(V1^2) +  I(V2^2)"
\donttest{design(treatments,blocks,treatments_model=model,searches=5)}
 
# Plackett and Burman design for eleven 2-level factors in 12 runs 
GF = expand.grid(F1 = factor(1:2,labels=c("a","b")), F2 = factor(1:2,labels=c("a","b")), 
                 F3 = factor(1:2,labels=c("a","b")), F4 = factor(1:2,labels=c("a","b")),
                 F5 = factor(1:2,labels=c("a","b")), F6 = factor(1:2,labels=c("a","b")),
                 F7 = factor(1:2,labels=c("a","b")), F8 = factor(1:2,labels=c("a","b")), 
                 F9 = factor(1:2,labels=c("a","b")), F10= factor(1:2,labels=c("a","b")), 
                 F11= factor(1:2,labels=c("a","b")) )
blocks=data.frame(main=gl(1,12))
model = "~ F1 + F2 + F3 + F4 + F5 + F6 + F7 + F8 + F9 + F10 + F11"
\donttest{D=design(GF,blocks,treatments_model=model,searches=25)
round(crossprod(scale(data.matrix(D$Design)[,-1])),6) }

## Factorial treatment designs defined by sequentially fitted factorial treatment models

## 4 varieties by 3 levels of N by 3 levels of K assuming degree-2 treatment
## interaction effects and two blocks of 12 plots
## the single stage model gives an unequal split for the replication of the four varieties
## which may be undesirable whereas the two stage model forces an equal split of 6 plots
## per variety. The single stage model appears slightly more efficient but
## in this example global D-optimality does not give the most suitable design structure. 
treatments = expand.grid(Variety = factor(1:4), N = 1:3, K = 1:3)
blocks=data.frame(main=gl(2,12))
treatments_model = " ~  (Variety + N + K)^2  + I(N^2) + I(K^2)"
design(treatments,blocks,treatments_model=treatments_model,searches=10) 
treatments_model = c(" ~ Variety" ," ~ Variety + (Variety + N + K)^2 + I(N^2) + I(K^2)")
design(treatments,blocks,treatments_model=treatments_model,searches=10)

}
\references{
Cochran W. G. & Cox G. M. (1957) Experimental Designs 2nd Edition John Wiley & Sons.
}
