% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BIOMOD_Modeling.R
\name{BIOMOD_Modeling}
\alias{BIOMOD_Modeling}
\title{Run a range of species distribution models}
\usage{
BIOMOD_Modeling(
  bm.format,
  modeling.id = as.character(format(Sys.time(), "\%s")),
  models = c("GLM", "GBM", "GAM", "CTA", "ANN", "SRE", "FDA", "MARS", "RF",
    "MAXENT.Phillips", "MAXENT.Phillips.2"),
  bm.options = NULL,
  nb.rep = 1,
  data.split.perc = 100,
  data.split.table = NULL,
  do.full.models = TRUE,
  weights = NULL,
  prevalence = NULL,
  metric.eval = c("KAPPA", "TSS", "ROC"),
  var.import = 0,
  save.output = TRUE,
  scale.models = FALSE,
  nb.cpu = 1,
  seed.val = NULL,
  do.progress = TRUE
)
}
\arguments{
\item{bm.format}{a \code{\link{BIOMOD.formated.data}} or \code{\link{BIOMOD.formated.data.PA}} 
object returned by the \code{\link{BIOMOD_FormatingData}} function}

\item{modeling.id}{a \code{character} corresponding to the name (ID) of the simulation set 
(\emph{a random number by default})}

\item{models}{a \code{vector} containing model names to be computed, must be among \code{GLM}, 
\code{GBM}, \code{GAM}, \code{CTA}, \code{ANN}, \code{SRE}, \code{FDA}, \code{MARS}, 
\code{RF}, \code{MAXENT.Phillips}, \code{MAXENT.Phillips.2}}

\item{bm.options}{a \code{\link{BIOMOD.models.options}} object returned by the
\code{\link{BIOMOD_ModelingOptions}} function}

\item{nb.rep}{an \code{integer} corresponding to the number of repetitions to be done for 
calibration/validation splitting}

\item{data.split.perc}{a \code{numeric} between \code{0} and \code{100} corresponding to the 
percentage of data used to calibrate the models (calibration/validation splitting)}

\item{data.split.table}{(\emph{optional, default} \code{NULL}) \cr 
A \code{matrix} or \code{data.frame} defining for each repetition (in columns) which 
observation lines should be used for models calibration (\code{TRUE}) and validation 
(\code{FALSE}) (see \code{\link{BIOMOD_CrossValidation}}) \cr (\emph{if specified, 
\code{nb.rep}, \code{data.split.perc} and \code{do.full.models} will be ignored})}

\item{do.full.models}{(\emph{optional, default} \code{TRUE}) \cr 
A \code{logical} value defining whether models calibrated and evaluated over the whole 
dataset should be computed or not}

\item{weights}{(\emph{optional, default} \code{NULL}) \cr 
A \code{vector} of \code{numeric} values corresponding to observation weights (one per 
observation, see Details)}

\item{prevalence}{(\emph{optional, default} \code{NULL}) \cr 
A \code{numeric} between \code{0} and \code{1} corresponding to the species prevalence to 
build '\emph{weighted response weights}' (see Details)}

\item{metric.eval}{a \code{vector} containing evaluation metric names to be used, must 
be among \code{ROC}, \code{TSS}, \code{KAPPA}, \code{ACCURACY}, \code{BIAS}, \code{POD}, 
\code{FAR}, \code{POFD}, \code{SR}, \code{CSI}, \code{ETS}, \code{HK}, \code{HSS}, \code{OR}, 
\code{ORSS}}

\item{var.import}{(\emph{optional, default} \code{NULL}) \cr 
An \code{integer} corresponding to the number of permutations to be done for each variable to 
estimate variable importance}

\item{save.output}{(\emph{optional, default} \code{TRUE}) \cr 
A \code{logical} value defining whether all outputs should be saved on hard drive or not 
(\emph{! strongly recommended !})}

\item{scale.models}{(\emph{optional, default} \code{FALSE}) \cr 
A \code{logical} value defining whether all models predictions should be scaled with a 
binomial GLM or not}

\item{nb.cpu}{(\emph{optional, default} \code{1}) \cr 
An \code{integer} value corresponding to the number of computing resources to be used to 
parallelize the single models computation}

\item{seed.val}{(\emph{optional, default} \code{NULL}) \cr 
An \code{integer} value corresponding to the new seed value to be set}

\item{do.progress}{(\emph{optional, default} \code{TRUE}) \cr 
A \code{logical} value defining whether the progress bar is to be rendered or not}
}
\value{
A \code{BIOMOD.models.out} object containing models outputs, or links to saved outputs. \cr
Models outputs are stored out of \R (for memory storage reasons) in 2 different folders 
created in the current working directory :
\enumerate{
  \item a \emph{models} folder, named after the \code{resp.name} argument of 
  \code{\link{BIOMOD_FormatingData}}, and containing all calibrated models for each 
  repetition and pseudo-absence run
  \item a \emph{hidden} folder, named \code{.BIOMOD_DATA}, and containing outputs related 
  files (original dataset, calibration lines, pseudo-absences selected, predictions, 
  variables importance, evaluation values...), that can be retrieved with 
  \href{https://biomodhub.github.io/biomod2/reference/getters.out.html}{\code{get_[...]}} or \code{\link{load}} functions, and used by other 
  \pkg{biomod2} functions, like \code{\link{BIOMOD_Projection}} or 
  \code{\link{BIOMOD_EnsembleModeling}}
}
}
\description{
This function allows to calibrate and evaluate a range of modeling techniques 
for a given species distribution. The dataset can be split up for independent calibration and 
validation, and the predictive power of the different models can be estimated using a range 
of evaluation metrics (see Details).
}
\details{
\describe{
  \item{bm.format}{If you have decided to add pseudo absences to your original dataset (see 
  \code{\link{BIOMOD_FormatingData}}), \cr \code{PA.nb.rep *(nb.rep + 1)} models will be 
  created.}
  
  \item{models}{The set of models to be calibrated on the data. 10 modeling techniques 
  are currently available :
  \itemize{
    \item \code{GLM} : Generalized Linear Model (\code{\link[stats]{glm}})
    \item \code{GAM} : Generalized Additive Model (\code{\link[gam]{gam}}, \code{\link[mgcv]{gam}} 
    or \code{\link[mgcv]{bam}}) \cr 
    (see \code{\link{BIOMOD_ModelingOptions} for details on algorithm selection})
    \item \code{GBM} : Generalized Boosting Model, or usually called Boosted Regression Trees 
    (\code{\link[gbm]{gbm}})
    \item \code{CTA} : Classification Tree Analysis (\code{\link[rpart]{rpart}})
    \item \code{ANN} : Artificial Neural Network (\code{\link[nnet]{nnet}})
    \item \code{SRE} : Surface Range Envelop or usually called BIOCLIM
    \item \code{FDA} : Flexible Discriminant Analysis (\code{\link[mda]{fda}})
    \item \code{MARS} : Multiple Adaptive Regression Splines (\code{\link[earth]{earth}})
    \item \code{RF} : Random Forest (\code{\link[randomForest]{randomForest}})
    \item \code{MAXENT.Phillips} : Maximum Entropy 
    (\url{https://biodiversityinformatics.amnh.org/open_source/maxent/})
    \item \code{MAXENT.Phillips.2} : Maximum Entropy (\code{\link[maxnet]{maxnet}})
  }}
  
  \item{nb.rep & data.split.perc}{
  \itemize{
    \item Most simple method in machine learning to calibrate and evaluate a model is to 
    split the original dataset in two, one to calibrate the model and the other one to 
    evaluate it. The \code{data.split.perc} argument defines the percentage of data that will be 
    randomly selected and used for the calibration part, the remaining data constituting the 
    evaluation part. This process is repeated \code{nb.rep} times, to be sure not to 
    include bias both in the modeling and evaluation parts.
    \item Other validation methods are also available to the user :
    \itemize{
      \item evaluation dataset can be directly given to the 
      \code{\link{BIOMOD_FormatingData}} function
      \item \code{data.split.table} argument can be used and obtained from the 
      \code{\link{BIOMOD_CrossValidation}} function
    }
  }}
  
  \item{weights & prevalence}{More or less weight can be given to some specific observations.
  \itemize{
    \item If \code{weights = prevalence = NULL}, each observation (presence or absence) will 
    have the same weight, no matter the total number of presences and absences.
    \item If \code{prevalence = 0.5}, presences and absences will be weighted equally 
    (\emph{i.e. the weighted sum of presences equals the weighted sum of absences}). 
    \item If \code{prevalence} is set below (\emph{above}) \code{0.5}, more weight will be 
    given to absences (\emph{presences}).
    \item If \code{weights} is defined, \code{prevalence} argument will be ignored, and each 
    observation will have its own weight.
    \item If pseudo-absences have been generated (\code{PA.nb.rep > 0} in 
    \code{\link{BIOMOD_FormatingData}}), weights are by default calculated such that 
    \code{prevalence = 0.5}. \emph{Automatically created \code{weights} will be \code{integer} 
    values to prevent some modeling issues.}
  }}

  \item{metric.eval}{
  \itemize{
    \item \code{ROC} : Relative Operating Characteristic
    \item \code{KAPPA} : Cohen's Kappa (Heidke skill score)
    \item \code{TSS} : True kill statistic (Hanssen and Kuipers discriminant, Peirce's skill 
    score)
    \item \code{FAR} : False alarm ratio
    \item \code{SR} : Success ratio
    \item \code{ACCURANCY} : Accuracy (fraction correct)
    \item \code{BIAS} : Bias score (frequency bias)
    \item \code{POD} : Probability of detection (hit rate)
    \item \code{CSI} : Critical success index (threat score)
    \item \code{ETS} : Equitable threat score (Gilbert skill score)
  }
  Optimal value of each method can be obtained with the \code{\link{get_optim_value}} 
  function. Several evaluation metrics can be selected. \emph{Please refer to the 
  \href{https://www.cawcr.gov.au/projects/verification/}{CAWRC website (section "Methods for dichotomous forecasts")} 
  to get detailed description of each metric.}
  }
  
  \item{save.output}{\emph{If this argument is set to \code{FALSE}, it may prevent the evaluation 
  of the ensemble models (see \code{\link{BIOMOD_EnsembleModeling}}) in further steps. Strong 
  recommandation is to keep \code{save.output = TRUE}, even if it requires to have some free 
  space onto the hard drive.}
  }
  
  \item{scale.models}{\bold{This parameter is quite experimental and it is recommended 
  not to use it. It may lead to reduction in projection scale amplitude.} Some categorical 
  models always have to be scaled (\code{FDA}, \code{ANN}), but it may be interesting to 
  scale all computed models to ensure comparable predictions (\code{0-1000} range). It might 
  be particularly useful when doing ensemble forecasting to remove the scale prediction effect 
  (\emph{the more extended projections are, the more they influence ensemble forecasting 
  results}).
  }
  
  \item{do.full.models}{Building models with all available information may be useful in some 
  particular cases (\emph{e.g. rare species with few presences points}). But calibration and 
  evaluation datasets will be the same, which might lead to over-optimistic evaluation scores.
  }
}
}
\examples{

# Load species occurrences (6 species available)
myFile <- system.file('external/species/mammals_table.csv', package = 'biomod2')
DataSpecies <- read.csv(myFile, row.names = 1)
head(DataSpecies)

# Select the name of the studied species
myRespName <- 'GuloGulo'

# Get corresponding presence/absence data
myResp <- as.numeric(DataSpecies[, myRespName])

# Get corresponding XY coordinates
myRespXY <- DataSpecies[, c('X_WGS84', 'Y_WGS84')]

# Load environmental variables extracted from BIOCLIM (bio_3, bio_4, bio_7, bio_11 & bio_12)
myFiles <- paste0('external/bioclim/current/bio', c(3, 4, 7, 11, 12), '.grd')
myExpl <- raster::stack(system.file(myFiles, package = 'biomod2'))

\dontshow{
myExtent <- raster::extent(0,30,45,70)
myExpl <- raster::stack(raster::crop(myExpl, myExtent))
}

# ---------------------------------------------------------------
# Format Data with true absences
myBiomodData <- BIOMOD_FormatingData(resp.var = myResp,
                                     expl.var = myExpl,
                                     resp.xy = myRespXY,
                                     resp.name = myRespName)

# Create default modeling options
myBiomodOptions <- BIOMOD_ModelingOptions()


# ---------------------------------------------------------------
# Model single models
myBiomodModelOut <- BIOMOD_Modeling(bm.format = myBiomodData,
                                    modeling.id = 'AllModels',
                                    models = c('RF', 'GLM'),
                                    bm.options = myBiomodOptions,
                                    nb.rep = 2,
                                    data.split.perc = 80,
                                    metric.eval = c('TSS','ROC'),
                                    var.import = 2,
                                    do.full.models = FALSE,
                                    seed.val = 42)
myBiomodModelOut

# Get evaluation scores & variables importance
get_evaluations(myBiomodModelOut)
get_variables_importance(myBiomodModelOut, as.data.frame = TRUE)

# Represent evaluation scores 
bm_PlotEvalMean(bm.out = myBiomodModelOut)
bm_PlotEvalBoxplot(bm.out = myBiomodModelOut, group.by = c('algo', 'run'))

# Represent variables importance 
# bm_PlotVarImpBoxplot(bm.out = myBiomodModelOut, group.by = c('expl.var', 'algo', 'algo'))
# bm_PlotVarImpBoxplot(bm.out = myBiomodModelOut, group.by = c('expl.var', 'algo', 'dataset'))
# bm_PlotVarImpBoxplot(bm.out = myBiomodModelOut, group.by = c('algo', 'expl.var', 'dataset'))

# Represent response curves 
# bm_PlotResponseCurves(bm.out = myBiomodModelOut, 
#                       models.chosen = get_built_models(myBiomodModelOut)[c(1:2)],
#                       fixed.var = 'median')
# bm_PlotResponseCurves(bm.out = myBiomodModelOut, 
#                       models.chosen = get_built_models(myBiomodModelOut)[c(1:2)],
#                       fixed.var = 'min')
# bm_PlotResponseCurves(bm.out = myBiomodModelOut, 
#                       models.chosen = get_built_models(myBiomodModelOut)[3],
#                       fixed.var = 'median',
#                       do.bivariate = TRUE)


}
\seealso{
\code{\link[stats]{glm}}, \code{\link[gam]{gam}}, \code{\link[mgcv]{gam}}, 
\code{\link[mgcv]{bam}}, \code{\link[gbm]{gbm}}, \code{\link[rpart]{rpart}}, 
code{\link[nnet]{nnet}}, \code{\link[mda]{fda}}, \code{\link[earth]{earth}}, 
\code{\link[randomForest]{randomForest}}, \code{\link[maxnet]{maxnet}},
\code{\link{BIOMOD_FormatingData}}, \code{\link{BIOMOD_ModelingOptions}}, 
\code{\link{BIOMOD_CrossValidation}}, \code{ \link{bm_VariablesImportance}}, 
\code{\link{BIOMOD_Projection}}, \code{\link{BIOMOD_EnsembleModeling}},
\code{\link{bm_PlotEvalMean}}, \code{\link{bm_PlotEvalBoxplot}}, 
\code{\link{bm_PlotVarImpBoxplot}}, \code{\link{bm_PlotResponseCurves}}

Other Main functions: 
\code{\link{BIOMOD_CrossValidation}()},
\code{\link{BIOMOD_EnsembleForecasting}()},
\code{\link{BIOMOD_EnsembleModeling}()},
\code{\link{BIOMOD_FormatingData}()},
\code{\link{BIOMOD_LoadModels}()},
\code{\link{BIOMOD_ModelingOptions}()},
\code{\link{BIOMOD_PresenceOnly}()},
\code{\link{BIOMOD_Projection}()},
\code{\link{BIOMOD_RangeSize}()},
\code{\link{BIOMOD_Tuning}()}
}
\author{
Wilfried Thuiller, Damien Georges, Robin Engler
}
\concept{Main functions}
\keyword{models}
\keyword{multivariate}
\keyword{nonlinear}
\keyword{nonparametric}
\keyword{regression}
\keyword{tree}
