#' Compose BioCompute Object - Top Level Fields (v1.3.0)
#'
#' @param provenance Provenance domain
#' @param usability Usability domain
#' @param extension Extension domain
#' @param description Description domain
#' @param execution Execution domain
#' @param parametric Parametric domain
#' @param io I/O domain
#' @param error Error domain
#' @param bco_id BioCompute Object identifier
#' (\href{https://github.com/biocompute-objects/BCO_Specification/blob/main/docs/top-level.md#202-biocompute-object-identifier-object_id}{definition}).
#' If \code{NULL}, will use a UUID generated by \code{\link{generate_id}}.
#'
#' @return A vector of top level fields
#'
#' @importFrom uuid UUIDgenerate
#'
#' @rdname compose_tlf
#' @export compose_tlf_v1.3.0
#'
#' @examples
#' compose_tlf(
#'   compose_provenance(), compose_usability(), compose_extension(),
#'   compose_description(), compose_execution(), compose_parametric(),
#'   compose_io(), compose_error()
#' ) %>% convert_json()
compose_tlf_v1.3.0 <-
  function(
           provenance, usability, extension, description,
           execution, parametric, io, error, bco_id = NULL) {
    bco_spec_version <- "https://w3id.org/biocompute/1.3.0/"
    if (is.null(bco_id)) bco_id <- generate_id("sevenbridges")

    lst <- list(
      "provenance_domain" = provenance,
      "usability_domain" = usability,
      "extension_domain" = extension,
      "description_domain" = description,
      "execution_domain" = execution,
      "parametric_domain" = parametric,
      "io_domain" = io,
      "error_domain" = error
    )

    json <- convert_json(lst)
    checksum <- digest::digest(json, algo = "sha256")

    domain <- c(
      "bco_spec_version" = bco_spec_version,
      "bco_id" = bco_id,
      "checksum" = checksum
    )
    domain
  }

#' @rdname compose_tlf
#' @export compose_tlf
compose_tlf <- compose_tlf_v1.3.0
