% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/relation.r
\name{relation}
\alias{relation}
\title{Relation vectors}
\usage{
relation(relations, attrs_order)
}
\arguments{
\item{relations}{a named list of relations, in the form of two-element lists:
the first element contains a data frame, where the column names are the
attributes in the associated schema, and the second element contains a list
of character vectors, each representing a candidate key.}

\item{attrs_order}{a character vector, giving the names of all attributes.
These need not be present in \code{schemas}, but all attributes in
\code{schemas} must be present in \code{attrs_order}.}
}
\value{
A \code{relation} object, containing the list given in
\code{relations}, with \code{attrs_order} stored in an attribute of the
same name. Relation schemas are returned with their keys' attributes sorted
according to the attribute order in \code{attrs_order}, and the keys then
sorted by priority order. Attributes in the data frame are also sorted,
first by order of appearance in the sorted keys, then by order in
\code{attrs_order} for non-prime attributes.
}
\description{
Creates a set of relation schemas, including the relation's attributes and
candidate keys.
}
\details{
Relation vectors are unlikely to be needed by the user directly, since they
are essentially \code{\link{database}} objects that can't have foreign key
references. They are mostly used to mirror the use of the vector-like
\code{\link{relation_schema}} class for the \code{\link{database_schema}}
class to be a wrapper around. This makes creating a \code{\link{database}}
from a \code{\link{relation_schema}} a two-step process, where the two steps
can be done in either order: creation with \code{\link{create}} and
\code{\link{insert}}, and adding references with
\code{\link{database_schema}} or \code{\link{database}}.

Duplicate schemas, after ordering by attribute, are allowed, and can be
removed with \code{\link{unique}}.

When several sets of relation schemas are concatenated, their
\code{attrs_order} attributes are merged, so as to preserve all of the original
attribute orders, if possible. If this is not possible, because the orderings
disagree, then the returned value of the \code{attrs_order} attribute is their
union instead.
}
\examples{
rels <- relation(
  list(
    a = list(
      df = data.frame(a = logical(), b = logical()),
      keys = list("a")
    ),
    b = list(
      df = data.frame(b = logical(), c = logical()),
      keys = list("b", "c")
    )
  ),
  attrs_order = c("a", "b", "c", "d")
)
print(rels)
records(rels)
attrs(rels)
stopifnot(identical(
  attrs(rels),
  lapply(records(rels), names)
))
keys(rels)
attrs_order(rels)
names(rels)

# inserting data
insert(rels, data.frame(a = 1L, b = 2L, c = 3L, d = 4L))
# data is only inserted into relations where all columns are given...
insert(rels, data.frame(a = 1L, b = 2L, c = 3L))
# and that are listed in relations argument
insert(
  rels,
  data.frame(a = 1L, b = 2L, c = 3L, d = 4L),
  relations = "a"
)

# vector operations
rels2 <- relation(
  list(
    e = list(
      df = data.frame(a = logical(), e = logical()),
      keys = list("e")
    )
  ),
  attrs_order = c("a", "e")
)
c(rels, rels2) # attrs_order attributes are merged
unique(c(rels, rels))

# subsetting
rels[1]
rels[c(1, 2, 1)]
stopifnot(identical(rels[[1]], rels[1]))

# reassignment
rels3 <- rels
rels3[2] <- relation(
  list(
    d = list(
      df = data.frame(d = logical(), c = logical()),
      keys = list("d")
    )
  ),
  attrs_order(rels3)
)
print(rels3) # note the relation's name doesn't change
# names(rels3)[2] <- "d" # this would change the name
keys(rels3)[[2]] <- list(character()) # removing keys first...
# for a relation_schema, we could then change the attrs for
# the second relation. For a created relation, this is not
# allowed.
\dontrun{
  attrs(rels3)[[2]] <- c("b", "c")
  names(records(rels3)[[2]]) <- c("b", "c")
}

# changing appearance priority for attributes
rels4 <- rels
attrs_order(rels4) <- c("d", "c", "b", "a")
print(rels4)

# reconstructing from components
rels_recon <- relation(
  Map(list, df = records(rels), keys = keys(rels)),
  attrs_order(rels)
)
stopifnot(identical(rels_recon, rels))

# can be a data frame column
data.frame(id = 1:2, relation = rels)
}
\seealso{
\code{\link{records}}, \code{\link{attrs}}, \code{\link{keys}}, and
\code{\link{attrs_order}} for extracting parts of the information in a
\code{relation_schema}; \code{\link{gv}} for converting the schema into
Graphviz code; \code{\link{rename_attrs}} for renaming the attributes in
\code{attrs_order}.
}
