#' @title Generic Summary Function for Class \code{ata}
#' @author Gulsah Gurkan (gurkangulsah@gmail.com), Michael Chajewski (mchajewski@hotmail.com)
#' @description Default summary function for output objects of class \code{ata}. The function provides a brief summary of the ATA form in text, and provides a binary table of constraint success.
#' @keywords ata summary "ata summary" test_form "test form"
#' @usage \\method{summary}{ata}(object, ...)
#' @method summary ata
#' @aliases summary summary.ata
#' @param object An output object of class \code{ata} generated by either \code{wdm()} or \code{atalp()} from the package.
#' @param ... Additional arguments affecting the summary produced.
#' @return The function returns a statement summarizing the evaluation of the assembled test form. Additionally, the function will return a pattern matrix for the test form constraints if assigned to an object.
#' \item{statement}{A summary of items (and/or item sets) in the test form and the overview of constraint success.}
#' \item{pattern}{A matrix of constraints by a classification if the additive constraints are below, at or above the constraint specific user provided bounds. This matrix, only returned if \code{summary.ata} is assigned to an object, will always demonstrate meeting all criteria for \code{atalp} test forms as all criteria have to be met to obtain a feasible solution.}
#' @export
summary.ata <- function(object, # ata class input
                        ...){   # Additional arguments to be passed to the function
  
  # ------------------------- #
  # Screening ata class input #
  # ------------------------- #
  
  # ata class object is missing (essential) elements.
  if(is.null(object[c("evaluation","final_ids")])){
    stop("The ata object input is missing elements.")
  }

  # ------------- #
  # Summary table #
  # ------------- #
  
  # Create a binary table based on achieved constraint values.
  Below <- as.numeric(ifelse(object$evaluation[3,-c(1,2)] < object$evaluation[1,-c(1,2)],1,0))
  Above <- as.numeric(ifelse(object$evaluation[3,-c(1,2)] > object$evaluation[2,-c(1,2)],1,0))
  Within <- as.numeric(ifelse(object$evaluation[1,-c(1,2)] <= object$evaluation[3,-c(1,2)] & object$evaluation[3,-c(1,2)] <= object$evaluation[2,-c(1,2)],1,0))
  success <- as.data.frame(t(rbind(Below,Within,Above)))
  Constraint <- names(object$evaluation[,-c(1,2)])
  success <- cbind(Constraint,success)
  
  # -------------------- #
  # Brief summary print  #
  # -------------------- #
  
  # Number of items (and sets) in the ATA form.
  if(attr(object,"aggregated")){
    cat(paste("ATA was formed by",as.character(length(object$final_ids)),"items from",length(object$final_setids),"sets."))
  }else{
    cat(paste("ATA was formed by",as.character(length(object$final_ids)),"items."))
  }
  
  # ATA method.
  cat(paste("\nMethod:", attr(object,"method")))
  
  # Achieved constraints.
  cat(paste("\n\n",sum(success$Within)," / ",dim(object$evaluation[,-c(1,2)])[2]," constraints within the defined bounds.\n", sep = ""))
  
  # If refined(wdm), number of items changed.
  if(attr(object,"method") %in% c("wdm") & attr(object,"refined")){
    cat(paste(sum(!object$final_ids %in% object$initial_ids),"item(s) replaced after refinement.\n"))
  }
  
  # If permutated (atalp), report.
  if(attr(object,"method") %in% c("lp") & attr(object,"permutated")){
    cat(paste("Final item selection was refined based on permutated solutions.\n"))
  }
  
  # Report if data was aggregated.
  if(attr(object,"aggregated")){
    cat("Data were aggregated by user-defined set_id before ATA was formed.\n")
  }
  
  # Report runtime.
  cat("\nATA form was created in", attr(object,"runtime"),units(attr(object,"runtime")),"\n")
  
  # Return table if assigned.
  invisible(success)
  
}


