% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utils.R
\name{check_one_logical}
\alias{check_one_logical}
\title{check_one_logical}
\description{
Use this function to validate parameters.
You're never assured that provided parameters from users are the right
type, or length. This may be the case with your own code as well,
if you have undetected bugs in your code.

This function helps prevent unpredictable behaviour coming from
bad parameters.

It checks the size of vectors, and the type of values.
If the parameter is not the good type or length, the program stops
with an explanatory error.
}
\examples{

## here is a simple utility function we will use in this example.
## It is not important
show_last_error <- function(error) {
  dump.frames()
  message(base::attr(last.dump, "error.message"))
}

## The example really starts here
## we have a simple function like this:
custom_message <- function(text) {
  message(sprintf("Message: \%s", text))
}

## this function needs to have a character vector as first
## parameter.
## So, to validate the parameter, we could write:
custom_message <- function(text) {
  check_parameter_type(text, "character")
  message(base::sprintf("Message: \%s", text))
}
tryCatch(custom_message(42), error = show_last_error)


## this function needs to have a vector of length 1.
## So, to validate the parameter, we could write:
custom_message <- function(text) {
  check_parameter_type(text, "character")
  check_parameter_length(text, 1)
  message(base::sprintf("Message: \%s", text))
}
tryCatch(custom_message(c("uwu", "owo")), error = show_last_error)


## Or, to be more concise:
custom_message <- function(text) {
  check_param_type_n_length(text, "character", 1)
  message(base::sprintf("Message: \%s", text))
}
tryCatch(custom_message(c("uwu", "owo")), error = show_last_error)
tryCatch(custom_message(42), error = show_last_error)


## Let's say the text can be 1 or more elements, and can be null.
custom_message <- function(text) {
  check_param_type_n_length(
    text,
    expected_type = "character",
    or_null = TRUE,
    expected_size = 1,
    or_more = TRUE
  )
  message(paste0(base::sprintf("Message: \%s", text), collapse = "\n"))
}
tryCatch(custom_message(c(42, 43)), error = show_last_error)
tryCatch(custom_message(NULL), error = show_last_error)
## no error, because or_null is TRUE
tryCatch(custom_message(character(0)), error = show_last_error)
tryCatch(custom_message(c("uwu", ":3")), error = show_last_error)
## no error, because or_more is TRUE

## With a function that has a lot of parameters, it may be usefull to
## provide the parameter's number. And, because it becomes very long
## to test all those parameters, we will use shortcuts functions
write_msg <- function(
  text,
  font = "owo",
  font_size = 16,
  italic = FALSE,
  bold = FALSE
) {
  check_one_character(text, nth = "1st")
  check_one_character(font, nth = "2nd")
  check_one_numeric(font_size, nth = "3rd")
  check_one_logical(italic, nth = "before last")
  check_one_logical(bold, nth = "last")
  message(paste0(base::sprintf("Message: \%s", text), collapse = "\n"))
}
tryCatch(write_msg(text = 42, "font", 16), error = show_last_error)
tryCatch(write_msg("uwu", font = 1, 16), error = show_last_error)
tryCatch(write_msg("uwu", font_size = "16"), error = show_last_error)
tryCatch(write_msg("uwu", italic = "FALSE"), error = show_last_error)
tryCatch(write_msg("uwu", bold = "FALSE"), error = show_last_error)

}
\seealso{
\link{check_parameter_type},\link{check_parameter_length}

\link{check_one_integer},\link{check_one_logical},\link{check_one_numeric}

\link{check_one_complex},\link{check_one_character}
}
\author{
L.Pavot
}
