% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/compute-GVA-functions.R
\name{compute_GVA}
\alias{compute_GVA}
\title{Compute the Full-Covariance Gaussian VB Empirical Likelihood Posterior}
\usage{
compute_GVA(
  mu,
  C,
  h,
  delthh,
  delth_logpi,
  z,
  lam0,
  rho,
  elip,
  a,
  iters,
  iters2,
  fullCpp,
  verbosity
)
}
\arguments{
\item{mu}{Column vector, initial value of Gaussian VB mean}

\item{C}{Lower Triangular Matrix, initial value of Gaussian VB Cholesky}

\item{h}{User-defined moment-condition function, outputs a k x 1 matrix containing the kth row of h. Function must take two arguments: zi and theta for h(zi,th)}

\item{delthh}{User defined function, outputs k x p Jacobian matrix of h(zi,th) with respect to theta}

\item{delth_logpi}{User-defined function, outputs p x 1 matrix, derivative of log prior function}

\item{z}{Data matrix, n-1 x d matrix}

\item{lam0}{Initial guess for lambda, k x 1 matrix}

\item{rho}{Scalar (between 0 to 1, reccomended to be close to 1) ADADELTA accumulation constant}

\item{elip}{Scalar numeric stability constant. Should be specified with a small value}

\item{a}{Scalar AEL constant, must be >0, small values recommended}

\item{iters}{Number of iterations for GVA (default:10,000)}

\item{iters2}{Number of iterations for Log AEL (default:500)}

\item{fullCpp}{Bool whether to calculate the main section in cpp (TRUE) or only partially (FALSE, doing all the AEL calculations in R before handing values to cpp) (default: TRUE)}

\item{verbosity}{Integer for how often to print updates on current iteration number (default:500)}
}
\value{
A list containing:  \enumerate{
\item mu_FC: VB Posterior Mean at final iteration. A vector of
size p x 1
\item C_FC: VB Posterior Variance-Covariance (Cholesky) at
final iteration. A lower-triangular matrix of size p x p
\item mu_FC_arr: VB Posterior Mean for each iteration. A matrix
of size p x (iters + 1)
\item C_FC_arr: VB Posterior Variance-Covariance (Cholesky) for
each iteration. An array of size p x p x (iters + 1)
}
}
\description{
Requires a given data set, moment conditions and parameter values and returns
a list of the final mean and variance-covariance along with an array of the
in-between calculations at each iteration for analysis of convergence
}
\examples{
set.seed(1)
x    <- runif(30, min = -5, max = 5)
elip <- rnorm(30, mean = 0, sd = 1)
y    <- 0.75 - x + elip
lam0 <- matrix(c(0,0), nrow = 2)
th   <- matrix(c(0.8277, -1.0050), nrow = 2)
a    <- 0.00001
z    <- cbind(x, y)
h    <- function(z, th) {
    xi     <- z[1]
    yi     <- z[2]
    h_zith <- c(yi - th[1] - th[2] * xi, xi*(yi - th[1] - th[2] * xi))
    matrix(h_zith, nrow = 2)
}

delthh <- function(z, th) {
    xi <- z[1]
    matrix(c(-1, -xi, -xi, -xi^2), 2, 2)
}

n     <- 31
reslm <- lm(y ~ x)
mu    <- matrix(unname(reslm$coefficients),2,1)
C_0   <- unname(t(chol(vcov(reslm))))

delth_logpi <- function(theta) { -0.0001 * mu }
elip        <- 10^-5
iters       <- 10
iters2      <- 50
rho         <- 0.9

# -----------------------------
# Main
# -----------------------------
ansGVARcppHalf <-compute_GVA(mu, C_0, h, delthh, delth_logpi, z, lam0, 
rho, elip, a, iters, iters2, fullCpp = FALSE)
ansGVARcppPure <-compute_GVA(mu, C_0, h, delthh, delth_logpi, z, lam0, 
rho, elip, a, iters, iters2, fullCpp = TRUE)

}
\references{
Yu, W., & Bondell, H. D. (2023). Variational Bayes for Fast and
Accurate Empirical Likelihood Inference. Journal of the American Statistical
Association, 1–13. \doi{doi:10.1080/01621459.2023.2169701}
}
\author{
Wei Chang Yu, Jeremy Lim
}
