% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/column_headers.R
\name{add_column_headers}
\alias{add_column_headers}
\title{Attach column headers to a Tplyr output}
\usage{
add_column_headers(.data, s, header_n = NULL)
}
\arguments{
\item{.data}{The data.frame/tibble on which the headers shall be attached}

\item{s}{The text containing the intended header string}

\item{header_n}{A header_n or generic data.frame to use for binding count values.
This is required if you are using the token replacement.}
}
\value{
A data.frame with the processed header string elements attached as the top rows
}
\description{
When working with 'huxtable' tables, column headers can be controlled as if they are rows in the data frame.
\code{add_column_headers} eases the process of introducing these headers.
}
\details{
Headers are created by providing a single string. Columns are specified by delimitting each header with a '|' symbol.
Instead of specifying the destination of each header, \code{add_column_headers} assumes that you have organized the columns
of your data frame before hand. This means that after you use \code{Tplyr::build()}, if you'd like to reorganize the
default column order (which is simply alphabetical), simply pass the build output to a \code{dplyr::select} or \code{dplyr::relocate}
statement before passing into \code{add_column_headers}.

Spanning headers are also supported. A spanning header is an overarching header that sits across multiple columns.
Spanning headers are introduced to \code{add_column_header} by providing the spanner text (i.e. the text that
you'd like to sit in the top row), and then the spanned text (the bottom row) within curly brackets ('\{\}). For example,
take the iris dataset. We have the names:

\code{"Sepal.Length" "Sepal.Width"  "Petal.Length" "Petal.Width"  "Species"}

If we wanted to provide a header string for this dataset, with spanners to help with categorization of
the variables, we could provide the following string:

\code{"Sepal {Length | Width} | Petal {Length | Width} | Species"}
}
\section{Important note}{

Make sure you are aware of the order of your variables prior to passing in to \code{add_column_headers}. The only requirement
is that the number of column match. The rest is up to you.
}

\section{Development notes}{

There are a few features of \code{add_column_header} that are intended but not yet supported:
\itemize{
\item{Nested spanners are not yet supported. Only a spanning row and a bottom row can currently be created}
\item{Different delimiters and indicators for a spanned group may be used in the future. The current choices were intuitive,
but based on feedback it could be determined that less common characters may be necessary.}
}
}

\section{Token Replacement}{

This function has support for reading values from the header_n object in a Tplyr table
and adding them in the column headers. Note: The order of the parameters
passed in the token is important. They should be first the treatment variable
then any \code{cols} variables in the order they were passed in the table construction.

Use a double asterisk "**" at the begining to start the token and another
double asterisk to close it. You can separate column parameters in the token
with a single underscore. For example, **group1_flag2_param3** will pull the count
from the header_n binding for group1 in the \code{treat_var}, flag2 in the first \code{cols}
argument, and param3 in the second \code{cols} argument.

You can pass fewer arguments in the token to get the sum of multiple columns.
For example, **group1** would get the sum of the group1 treat_var,
and all cols from the header_n.
}

\examples{
# Load in pipe
library(magrittr)
library(dplyr)
header_string <- "Sepal {Length | Width} | Petal {Length | Width} | Species"

iris2 <- iris \%>\%
  mutate_all(as.character)

iris2 \%>\% add_column_headers(header_string)

# Example with counts
mtcars2 <- mtcars \%>\%
  mutate_all(as.character)

t <- tplyr_table(mtcars2, vs, cols = am) \%>\%
  add_layer(
    group_count(cyl)
  )

b_t <- build(t) \%>\%
  mutate_all(as.character)

count_string <- paste0(" | V N=**0** {auto N=**0_0** | man N=**0_1**} |",
                       " S N=**1** {auto N=**1_0** | man N=**1_1**} | | ")

add_column_headers(b_t, count_string, header_n(t))
}
