#' @title 'GTSplot'
#' @description General Plotly method only working on time series data
#'
#' @param tsdata A vector or a data frame contains information of time series dataset(i.e. created by \code{ts} function)
#' @param Unit the unit of time for the time series data
#' @param NEWtitle title for this plot
#' @param Ylab label of Y axis
#' @param Xlab label of X axis
#' @param ts_name a vector contains names for each time line
#' @param COLO a vector contains colors for each time line
#' @param title_size size of the title
#'
#' @details
#' The function \code{TSplot} is based on package \code{plotly}. It applies \code{plot_ly} function to create
#' interactive plot for time series data(i.e data generated by function \code{ts}).
#' @return a plot result created by plot_ly() function
#'
#' @author SOCR team <\url{http://socr.umich.edu/people/}>
#' @export
#'
#' @import forecast plotly dcemriS4
#' @importFrom stats time
#'
#' @examples
#' require(forecast)
#' require(dcemriS4)
#' require(plotly)
#'
#' ## In the "fMRI" chapter, we have a 4-dimension dataset
#' ## with x,y,z and time dimension (dataset "fMRIVolume").
#' ## So we can settle x,y,and z to determine a vector of time series data.
#'
#' # You could find the raw "fMRIVolume" dataset on the SOCR website
#' \donttest{fMRIURL <- "http://socr.umich.edu/HTML5/BrainViewer/data/fMRI_FilteredData_4D.nii.gz"}
#' \donttest{fMRIFile <- file.path(tempdir(), "fMRI_FilteredData_4D.nii.gz")}
#' \donttest{download.file(fMRIURL, dest=fMRIFile, quiet=TRUE)}
#' \donttest{fMRIVolume <- readNIfTI(fMRIFile, reorient=FALSE)}
#'
#'
#' # Load three time series data(with a wrong format)
#' \donttest{xA_fMRI_1D_x20_y20_z11 <- fMRIVolume[20, 20, 11, ]}
#' \donttest{xB_fMRI_1D_x30_y30_z13 <- fMRIVolume[30, 30, 13, ]}
#' \donttest{xC_fMRI_1D_x40_y40_z12 <- fMRIVolume[40, 40, 12, ]}
#'
#' # Change this to time series data
#' TS1<-ts(xA_fMRI_1D_x20_y20_z11,start=0,frequency =1/3)
#' TS2<-ts(xB_fMRI_1D_x30_y30_z13,start=0,frequency =1/3)
#' TS3<-ts(xC_fMRI_1D_x40_y40_z12,start=0,frequency =1/3)
#'
#' # Package them into a data frame
#' TSDF<-data.frame(TS1,TS2,TS3)
#'
#' # Using this function to create plot
#' GTSplot(TSDF,Xlab="Time(second)",Unit="sec",ts_name=c("xA_fMRI_1D_x20_y20_z11",
#' "xB_fMRI_1D_x30_y30_z13","xC_fMRI_1D_x40_y40_z12"),
#'               COLO=c("green","red","blue"))
GTSplot<- function(tsdata,NEWtitle="Result",Ylab="Value",Xlab="Time",Unit=NULL
                  ,ts_name=NULL,title_size=10,COLO=NULL) {
  TSP<-plot_ly(type="scatter",mode="lines")
  for (i in 1:ncol(tsdata)) {
    tsd<-tsdata[,i]
    tsn<-ts_name[i]
    Color<-COLO[i]
    TSP<-add_trace(TSP,x=time(tsd),text=paste(time(tsd),Unit),type="scatter",mode="lines",
                   y=tsd,name=tsn,line=list(Color))
  }
  TSP<-TSP%>%
    layout(title = list(text = NEWtitle,font = list(family = "Times New Roman",size = title_size,color = "black" )),
           paper_bgcolor='rgb(255,255,255)', plot_bgcolor='rgb(229,229,229)',
           xaxis = list(title = Xlab,
                        gridcolor = 'rgb(255,255,255)',
                        showgrid = TRUE,
                        showline = FALSE,
                        showticklabels = TRUE,
                        tickcolor = 'rgb(127,127,127)',
                        ticks = 'outside',
                        zeroline = FALSE),
           yaxis = list(title = Ylab,
                        gridcolor = 'rgb(255,255,255)',
                        showgrid = TRUE,
                        showline = FALSE,
                        showticklabels = TRUE,
                        tickcolor = 'rgb(127,127,127)',
                        ticks = 'outside',
                        zeroline = FALSE))
  return(TSP)
}
