

#' Compute surrogacy measures for a binary surrogate and a time-to-event true endpoint in the meta-analytic multiple-trial setting.
#'
#'The function 'survbin()' fits the model for a binary surrogate and time-to-event true endpoint developed by Burzykowski et al. (2004) in the meta-analytic multiple-trial setting.
#'
#' @details
#'
#' # Model
#'
#' In the model developed by Burzykowski et al. (2004), a copula-based model is used for the true endpoint and a latent continuous variable, underlying the surrogate endpoint.
#' More specifically, the Plackett copula is used. The marginal model for the surrogate endpoint is a logistic regression model. For the true endpoint, the proportional hazard model is used.
#' The quality of the surrogate at the individual level can be evaluated by using the copula parameter Theta, which takes the form of a global odds ratio.
#' The quality of the surrogate at the trial level can be evaluated by considering the correlation coefficient between the estimated treatment effects, while adjusting for the estimation error.
#'
#' # Data Format
#'
#' The data frame must contains the following columns:
#'
#' * a column with the observed time-to-event (true endpoint)
#' * a column with the time-to-event indicator: 1 if true event is observed, 0 otherwise
#' * a column with the binary surrogate endpoint: 1 or 2
#' * a column with the treatment indicator: 0 or 1
#' * a column with the trial indicator
#' * a column with the center indicator. If there are no different centers within each trial, the center indicator is equal to the trial indicator
#' * a column with the patient indicator
#'
#' @references Burzykowski, T., Molenberghs, G., & Buyse, M. (2004). The validation of surrogate end points by using data from randomized clinical trials: a case-study in advanced colorectal cancer. Journal of the Royal Statistical Society Series A: Statistics in Society, 167(1), 103-124.
#'
#' @param data A data frame with the correct columns (See details).
#' @param true Observed time-to-event (true endpoint).
#' @param trueind Time-to-event indicator.
#' @param surrog Binary surrogate endpoint.
#' @param trt Treatment indicator.
#' @param center Center indicator (equal to trial if there are no different centers).
#' @param trial Trial indicator.
#' @param patientid Patient indicator.
#'
#' @return Returns an object of class "survbin" that can be used to evaluate surrogacy and contains the following elements:
#'
#' * Indiv.GlobalOdds: a data frame that contains the Global Odds and 95% confidence interval to evaluate surrogacy at the individual level.
#' * Trial.R2: a data frame that contains the correlation coefficient and 95% confidence interval to evaluate surrogacy at the trial level.
#' * EstTreatEffects: a data frame that contains the estimated treatment effects and sample size for each trial.
#'
#' @export
#'
#' @author Dries De Witte
#'
#' @examples
#' \dontrun{
#' data("colorectal")
#' fit_bin <- survbin(data = colorectal, true = surv, trueind = SURVIND, surrog = responder,
#'                    trt = TREAT, center = CENTER, trial = TRIAL, patientid = patientid)
#' print(fit_bin)
#' summary(fit_bin)
#' plot(fit_bin)
#' }
#'
survbin <- function(data, true, trueind, surrog,
                    trt, center, trial, patientid) {
  resp_est <- surv_est <- sample_size <- Center_ID <- p <- shape <- variable <- NULL
  dataset1xxy <- data
  dataset1xxy$surv <- data[[substitute(true)]]
  dataset1xxy$survind <- data[[substitute(trueind)]]
  dataset1xxy$suro <- data[[substitute(surrog)]]
  dataset1xxy$Z <- data[[substitute(trt)]]
  dataset1xxy$Center_ID <- data[[substitute(center)]]
  dataset1xxy$Trial_ID <- data[[substitute(trial)]]
  dataset1xxy$Pat_ID <- data[[substitute(patientid)]]

  assign_data <- function(dataset1xxy) {
    dataset1xxy <- dataset1xxy[order(dataset1xxy$Center_ID), ]
    centnum <- unique(dataset1xxy$Center_ID)
    centid <- list()

    for (i in seq_along(centnum)) {
      centid[[paste0("centid", sprintf("%02d", i))]] <- centnum[i]
    }

    c <- sprintf("%02d", length(centnum))

    for (i in 1:length(centnum)) {
      const_var <- paste0("const", sprintf("%02d", i))
      treat_var <- paste0("treat", sprintf("%02d", i))

      dataset1xxy[[const_var]] <- as.integer(dataset1xxy$Center_ID == centid[[paste0("centid", sprintf("%02d", i))]])
      dataset1xxy[[treat_var]] <- as.numeric(dataset1xxy[[const_var]]) * as.numeric(dataset1xxy[["Z"]])
    }

    return(dataset1xxy)
  }

  dataset1xxy <- assign_data(dataset1xxy)

  c <- length(unique(dataset1xxy$Center_ID))
  covariate_names <- paste0("treat", sprintf("%02d", 1:c))

  dataset1xxy$suro2 <- dataset1xxy$suro-1
  dataset1xxy$suro2_bis <- factor(dataset1xxy$suro2, levels = c("1", "0"))
  formula <- as.formula(paste("suro2_bis ~", paste(covariate_names, collapse = " + ")))

  model <- stats::glm(formula, data = dataset1xxy,
               family = binomial(link = "logit"))
  ests <- coef(model)
  estimS <- as.matrix(ests)

  t <- as.matrix(dataset1xxy$surv)
  delta <- as.vector(dataset1xxy$survind)
  s <- as.vector(dataset1xxy$suro)
  x <- as.matrix(dataset1xxy$Z)
  center <- as.vector(dataset1xxy$Center_ID)
  trial <- as.vector(dataset1xxy$Trial_ID)

  qq <- ncol(x)
  qq
  r <- 0
  K <- length(unique(s))

  cents <- unique(center)
  numcents <- length(cents)
  tri <- unique(trial)
  numtri <- length(tri)

  par0s <- estimS

  intercS <- par0s[1:(K - 1), ]
  efftrtS <- par0s[(K - 1 + r + 1):(K - 1 + r + numcents * qq), ]

  if (numtri > 1) {
    hulp <- matrix(0, nrow = numtri - 1, ncol = 1)
    hulp <- as.vector(hulp)
    param0s <- c(intercS, hulp, efftrtS)
  } else {
    param0s <- c(intercS, efftrtS)
  }

  estimate <- as.data.frame(param0s)
  dataset1xxy <- dataset1xxy[order(dataset1xxy$Center_ID),]
  model_results <- list()

  estt2 <- data.frame(center = numeric(),
                      intercept = numeric(),
                      treat_effect = numeric(),
                      scale = numeric(),
                      stringsAsFactors = FALSE)

  centers <- unique(dataset1xxy$Center_ID)

  for (i in seq_along(centers)) {
    center_data <- subset(dataset1xxy, Center_ID == centers[i])
    model <- survival::survreg(Surv(surv, survind) ~ Z, data = center_data, dist = "weibull")

    intercept <- coef(model)["(Intercept)"]
    treat_effect <- coef(model)["Z"]
    scale_param <- model$scale

    estt2 <- estt2 %>%
      tibble::add_row(center = as.numeric(centers[i]),
              intercept = intercept,
              treat_effect = treat_effect,
              scale = scale_param)
  }
  estt <- estt2
  estt$p <- 1 / estt$scale
  estt$shape <- -estt$intercept
  estt$trt <- -estt$treat_effect / estt$scale
  estt <- estt[, c("center", "p", "shape", "trt")]

  estimT <- estt %>%
    tidyr::pivot_longer(cols = c(p, shape, trt),
                 names_to = "variable",
                 values_to = "value") %>%
    dplyr::mutate(variable = dplyr::case_when(
      variable == "p" ~ "p",
      variable == "shape" ~ "shape",
      variable == "trt" ~ "trt"
    ))

  estimT <- estimT %>%
    arrange(variable, center)
  estimT <- estimT[, "value", drop = FALSE]
  estim <- estimT

  q1 <- function(u, v, theta) {
    quv <- sqrt((1 + (theta - 1) * (u + v))^2 - 4 * u * v * theta * (theta - 1))
    return(quv)
  }

  d1qth <- function(u, v, theta) {
    dq <- ((1 + (theta - 1) * (u + v)) * (u + v) - 2 * u * v * (2 * theta - 1)) / q1(u, v, theta)
    return(dq)
  }

  d1qu <- function(u, v, theta) {
    dq <- (theta - 1) * (1 + (theta - 1) * (u + v) - 2 * v * theta) / q1(u, v, theta)
    return(dq)
  }


  d1qv <- function(u, v, theta) {
    dq <- (theta - 1) * (1 + (theta - 1) * (u + v) - 2 * u * theta) / q1(u, v, theta)
    return(dq)
  }

  d2qth2 <- function(u, v, theta) {
    d2q <- ((u - v)^2 - d1qth(u, v, theta)^2) / q1(u, v, theta)
    return(d2q)
  }

  d2qu2 <- function(u, v, theta) {
    d2q <- ((theta - 1)^2 - d1qu(u, v, theta)^2) / q1(u, v, theta)
    return(d2q)
  }

  d2qv2 <- function(u, v, theta) {
    d2q <- ((theta - 1)^2 - d1qv(u, v, theta)^2) / q1(u, v, theta)
    return(d2q)
  }

  d2quv <- function(u, v, theta) {
    d2q <- -((theta - 1) * (theta + 1) + d1qu(u, v, theta) * d1qv(u, v, theta)) / q1(u, v, theta)
    return(d2q)
  }

  d2quth <- function(u, v, theta) {
    q <- q1(u, v, theta)
    d1qu <- d1qu(u, v, theta)
    d1qth <- d1qth(u, v, theta)
    d2q <- (theta - 1) * (u - v) / q + d1qu * (1 / (theta - 1) - d1qth / q)
    return(d2q)
  }

  d2quth(2, 3, 0.5)

  d2qvth <- function(u, v, theta) {
    q <- q1(u, v, theta)
    d1qv <- d1qv(u, v, theta)
    d1qth <- d1qth(u, v, theta)
    d2q <- (theta - 1) * (v - u) / q + d1qv * (1 / (theta - 1) - d1qth / q)
    return(d2q)
  }

  C <- function(u, v, theta) {
    if (theta != 1)
      Cuv <- (1 + (theta - 1) * (u + v) - q1(u, v, theta)) / (2 * (theta - 1))
    else
      Cuv <- u * v
    return(Cuv)
  }

  d1Cth <- function(u, v, theta) {
    if (theta != 1)
      dC <- (-C(u, v, theta) + (u + v - d1qth(u, v, theta)) / 2) / (theta - 1)
    else
      dC <- matrix(0, nrow(u), 1)
    return(dC)
  }

  d1Cv <- function(u, v, theta) {
    if (theta != 1)
      dC <- (1 - d1qv(u, v, theta) / (theta - 1)) / 2
    else
      dC <- u
    return(dC)
  }

  d1Cu <- function(u, v, theta) {
    if (theta != 1)
      dC <- (1 - d1qu(u, v, theta) / (theta - 1)) / 2
    else
      dC <- v
    return(dC)
  }

  d2Cu2 <- function(u, v, theta) {
    if (theta != 1)
      d2C <- -(theta - 1) * (1 - (d1qu(u, v, theta) / (theta - 1))^2) / (2 * q1(u, v, theta))
    else
      d2C <- matrix(0, nrow(u), 1)
    return(d2C)
  }

  d2Cv2 <- function(u, v, theta) {
    if (theta != 1)
      d2C <- -(theta - 1) * (1 - (d1qv(u, v, theta) / (theta - 1))^2) / (2 * q1(u, v, theta))
    else
      d2C <- matrix(0, nrow(u), 1)
    return(d2C)
  }

  d2Cth2 <- function(u, v, theta) {
    if (theta != 1)
      d2C <- -(2 * d1Cth(u, v, theta) + d2qth2(u, v, theta) / 2) / (theta - 1)
    else
      d2C <- matrix(0, nrow(u), 1)
    return(d2C)
  }

  d2Cuv <- function(u, v, theta) {
    if (theta != 1)
      d2C <- (theta + 1 + d1qu(u, v, theta) * d1qv(u, v, theta) / (theta - 1)) / (2 * q1(u, v, theta))
    else
      d2C <- matrix(1, nrow(u), 1)
    return(d2C)
  }

  d2Cuth <- function(u, v, theta) {
    if (theta != 1)
      d2C <- (v - u + d1qu(u, v, theta) * d1qth(u, v, theta) / (theta - 1)) / (2 * q1(u, v, theta))
    else
      d2C <- matrix(0, nrow(u), 1)
    return(d2C)
  }

  d2Cvth <- function(u, v, theta) {
    if (theta != 1)
      d2C <- (u - v + d1qv(u, v, theta) * d1qth(u, v, theta) / (theta - 1)) / (2 * q1(u, v, theta))
    else
      d2C <- matrix(0, nrow(u), 1)
    return(d2C)
  }

  d3Cu3 <- function(u, v, theta) {
    if (theta != 1) {
      q_val <- q1(u, v, theta)
      dqu <- d1qu(u, v, theta)
      d2qu2 <- d2qu2(u, v, theta)
      d2Cu2 <- d2Cu2(u, v, theta)
      d3C <- (-d2Cu2 + d2qu2 / (theta - 1)) * dqu / q_val
    } else {
      d3C <- matrix(0, nrow(u), 1)
    }
    return(d3C)
  }

  d3Cv3 <- function(u, v, theta) {
    if (theta != 1) {
      q_val <- q1(u, v, theta)
      dqv <- d1qv(u, v, theta)
      d2qv2 <- d2qv2(u, v, theta)
      d2Cv2 <- d2Cv2(u, v, theta)
      d3C <- (-d2Cv2 + d2qv2 / (theta - 1)) * dqv / q_val
    } else {
      d3C <- matrix(0, nrow(u), 1)
    }
    return(d3C)
  }

  d3Cu2v <- function(u, v, theta) {
    if (theta != 1) {
      q_val <- q1(u, v, theta)
      dqv <- d1qv(u, v, theta)
      dqu <- d1qu(u, v, theta)
      d2quv <- d2quv(u, v, theta)
      d2Cu2 <- d2Cu2(u, v, theta)
      d3C <- (-d2Cu2 * dqv + dqu * d2quv / (theta - 1)) / q_val
    } else {
      d3C <- matrix(0, nrow(u), 1)
    }
    return(d3C)
  }

  d3Cuv2 <- function(u, v, theta) {
    if (theta != 1) {
      q_val <- q1(u, v, theta)
      dqv <- d1qv(u, v, theta)
      dqu <- d1qu(u, v, theta)
      d2quv <- d2quv(u, v, theta)
      d2Cv2 <- d2Cv2(u, v, theta)
      d3C <- (-d2Cv2 * dqu + dqv * d2quv / (theta - 1)) / q_val
    } else {
      d3C <- matrix(0, nrow(u), 1)
    }
    return(d3C)
  }

  d3Cuth2 <- function(u, v, theta) {
    if (theta != 1) {
      q <- q1(u, v, theta)
      dqu <- d1qu(u, v, theta)
      dqth <- d1qth(u, v, theta)
      d2quth <- d2quth(u, v, theta)
      d2qth2 <- d2qth2(u, v, theta)
      d2Cuth <- d2Cuth(u, v, theta)
      d3C <- -d2Cuth * (dqth / q + 1 / (theta - 1)) + (dqu * d2qth2 + d2quth * dqth - (u - v)) / (2 * q * (theta - 1))
    } else {
      d3C <- matrix(0, nrow(u), 1)
    }
    return(d3C)
  }

  d3Cu2th <- function(u, v, theta) {
    if (theta != 1) {
      q <- q1(u, v, theta)
      dqu <- d1qu(u, v, theta)
      dqth <- d1qth(u, v, theta)
      d2quth <- d2quth(u, v, theta)
      d2Cu2 <- d2Cu2(u, v, theta)
      d3C <- d2Cu2 * (1 / (theta - 1) - dqth / q) + dqu * d2quth / (q * (theta - 1))
    } else {
      d3C <- matrix(0, nrow(u), 1)
    }
    return(d3C)
  }

  d3Cuvth <- function(u, v, theta) {
    if (theta != 1) {
      q <- q1(u, v, theta)
      dqu <- d1qu(u, v, theta)
      dqv <- d1qv(u, v, theta)
      dqth <- d1qth(u, v, theta)
      d2quth <- d2quth(u, v, theta)
      d2qvth <- d2qvth(u, v, theta)
      d2Cuv <- d2Cuv(u, v, theta)
      d3C <- -d2Cuv * (dqth / q + 1 / (theta - 1)) + (d2quth * dqv + d2qvth * dqu + 2 * theta) / (2 * q * (theta - 1))
    } else {
      d3C <- matrix(0, nrow(u), 1)
    }
    return(d3C)
  }

  cFt <- function(lambda, p, beta, t, s, zt, zs) {
    lt <- lambda * t
    Ft <- 1 - exp(-(lt^p) * exp(zt %*% beta))
    return(Ft)
  }

  pft <- function(lambda, p, beta, t, s, zt, zs) {
    lt <- lambda * t
    f <- lambda * p * (lt^(p - 1)) * exp(zt %*% beta) * exp(-(lt^p) * exp(zt %*% beta))
    return(f)
  }

  cFs <- function(alpha, t, s, zt, zs) {
    Fs <- exp(zs %*% alpha) / (1 + exp(zs %*% alpha))
    return(Fs)
  }

  xsect <- function(...) {
    args <- list(...)
    result <- Reduce(intersect, args)
    result <- sort(result)
    return(result)
  }

  design <- function(x) {
    unique_vals <- sort(unique(x))
    design_matrix <- matrix(0, nrow = length(x), ncol = length(unique_vals))

    for (i in 1:length(unique_vals)) {
      design_matrix[, i] <- as.numeric(x == unique_vals[i])
    }

    colnames(design_matrix) <- unique_vals
    return(design_matrix)
  }

  pgts <- function(theta, lambda, p, beta, alpha, t, s, zt, zs, x) {
    pft <- pft(lambda, p, beta, t, s, zt, zs)
    cFt <- cFt(lambda, p, beta, t, s, zt, zs)

    K <- length(unique(s))

    dCu <- NULL
    cFs <- NULL

    for (i in 1:(K-1)) {
      zero <- matrix(0, nrow = nrow(as.matrix(x)), ncol = K-1)

      zero[, i] <- 1

      zsi <- cbind(zero, matrix(1, nrow = nrow(as.matrix(x)), ncol = 1), x)

      cFsi <- cFs(alpha, t, s, zt, zsi)

      cFs <- cbind(cFs, cFsi)

      dCu <- cbind(dCu, d1Cu(cFt, cFsi, theta))

    }

    dCui <- cbind(dCu, matrix(1, nrow = nrow(dCu), ncol = 1))
    dCui_1 <- cbind(matrix(0, nrow = nrow(as.matrix(x)), ncol = 1), dCu)

    yresp <- xsect(1:K,unique(s))
    des <- matrix(0, nrow = length(s), ncol = K)

    if (length(yresp) < K) {
      des[cbind(1:length(s), match(s, yresp))] <- 1
    } else {
      des <- design(s)
    }

    g <- as.matrix(rowSums(des*((dCui - dCui_1) * as.vector(pft))))

    return(g)
  }

  cGts <- function(theta, lambda, p, beta, alpha, t, s, zt, zs, x) {
    cFt <- cFt(lambda, p, beta, t, s, zt, zs)
    K <- length(unique(s))

    cFs <- NULL
    Fts <- NULL

    for (i in 1:(K-1)) {
      zero <- matrix(0, nrow = nrow(as.matrix(x)), ncol = K-1)
      zero[, i] <- 1
      zsi <- cbind(zero, matrix(1, nrow = nrow(as.matrix(x)), ncol = 1), x)
      cFsi <- cFs(alpha, t, s, zt, zsi)
      cFs <- cbind(cFs, cFsi)
      Fts <- cbind(Fts, C(cFt, cFsi, theta))
    }

    diffs1 <- cbind(cFs, matrix(1, nrow = nrow(as.matrix(cFs)), ncol = 1))-cbind(matrix(0, nrow = nrow(as.matrix(cFs)), ncol = 1),cFs)
    diffs2 <- cbind(Fts, cFt) - cbind(matrix(0, nrow(as.matrix(Fts)), 1), Fts)

    yresp <- xsect(1:K,unique(s))
    des <- matrix(0, nrow = length(s), ncol = K)
    if (length(yresp) < K) {
      des[cbind(1:length(s), match(s, yresp))] <- 1
    } else {
      des <- design(s)
    }

    g <- as.matrix(rowSums(des * (diffs1 - diffs2)))

    #g=(des#(diffs1-diffs2))[,+];

    return(g)
  }

  loglik <- function(param) {
    t <- as.matrix(dataset1xxy$surv)
    delta <- as.vector(dataset1xxy$survind)
    s <- as.vector(dataset1xxy$suro)
    x <- as.matrix(dataset1xxy$Z)
    center <- as.vector(dataset1xxy$Center_ID)
    trial <- as.vector(dataset1xxy$Trial_ID)

    cents <- unique(center)
    numcents <- length(cents)
    tri <- unique(trial)
    numtri <- length(tri)
    K <- length(unique(s))

    zt <- x
    hulp <- design(s)
    des <- hulp[,1:(K-1)]
    zs <- as.matrix(cbind(des, matrix(1, length(x), 1), x))

    theta <- param[1]
    eta <- param[(1 + 2 * numtri + numcents + 1):(1 + 2 * numtri + numcents + K - 1)]
    lik <- 0
    for (i in 1:numcents) {
      ti <- t[center == cents[i]]
      si <- s[center == cents[i]]
      deltai <- delta[center == cents[i]]
      zti <- as.matrix(zt[center == cents[i], ])
      zsi <- as.matrix(zs[center == cents[i], ])
      xi <- x[center == cents[i], ]
      hulp <- as.numeric(unique(trial[center == cents[i]]))
      stud <- which(tri == hulp)
      pi <- param[1 + stud]
      lli <- param[1 + numtri + stud]
      efftrtT <- param[1 + 2 * numtri + i]
      betai <- as.matrix(cbind(efftrtT))
      if (stud == 1) {
        interc <- 0
      } else {
        interc <- param[1 + 2 * numtri + numcents + (K - 1) + stud - 1]
      }
      efftrtS <- param[1 + 2 * numtri + numcents + (K - 1) + numtri - 1 + i]
      alphai <- as.matrix(c(eta, interc, efftrtS))

      lambdai <- exp(lli)
      lti <- lambdai * t[center == cents[i], ]
      pgi <- pgts(theta, lambdai, pi, betai, alphai, ti, si, zti, zsi, xi)
      cGi <- cGts(theta, lambdai, pi, betai, alphai, ti, si, zti, zsi, xi)

      lik <- lik + sum(deltai * log(pgi) + (1 - deltai) * log(cGi))
    }

    return(lik)
  }

  initparm <- function(theta0) {
    s <- as.vector(dataset1xxy$suro)
    center <- as.vector(dataset1xxy$Center_ID)

    center <- unique(center)
    cents <- unique(center)
    numcents <- length(cents)
    K <- length(unique(s))

    param1 <- estim
    param2 <- estimate

    param0 <- cbind(theta0, t(param1), t(param2))

    return(param0)
  }

  initial_parameters <- initparm(5)

  negll <- function(param){
    loglike <- loglik(param)
    val <- -loglike
    return(val)
  }

  numgrad <- numDeriv::grad(x = initial_parameters, func = negll)

  suppressWarnings(opt_BFGS <- optimx::optimx(par = as.vector(initial_parameters), fn=negll, hessian = TRUE, method = "BFGS",
                     control = list(trace = 0, maxit=10000)))
  if(opt_BFGS$convcode != 0) {
    stop(paste("Optimization algorithm did not converge: convcode of optimx-function is equal to", opt_BFGS$convcode))
  }

  hessian_m <- attributes(opt_BFGS)$details["BFGS", "nhatend"][[1]]
  fisher_info <- solve(hessian_m, tol = 1e-35)
  prop_se <- sqrt(diag(fisher_info))
  coef_BFGS <- coef(opt_BFGS)["BFGS",]
  coef_se <- cbind(coef_BFGS, prop_se)

  endp1 <- coef_se[(2*numtri+1+1):(2*numtri+1+numcents), ]
  endp2 <- coef_se[(1+3*numtri+numcents+(K-1)-1+1):nrow(coef_se), ]

  weight <- matrix(0, nrow = numcents, ncol = 1)
  for (i in 1:numcents) {
    weight[i, ] <- nrow(as.matrix(which(center == cents[i])))
  }

  cova <- diag(fisher_info[(2*numtri+1+1):(2*numtri+1+numcents), (1+3*numtri+numcents+(K-1)-1+1):length(coef_BFGS)])

  memo <- cbind(cents, endp1, endp2, cova, weight)
  colnames(memo) <- c("center", "surv_est", "surv_se", "resp_est", "resp_se", "cova", "weight")
  memo <- as.data.frame(memo)

  #### INDIVIDUAL LEVEL ####
  lo <- coef_BFGS - qnorm(0.975) * prop_se
  up <- coef_BFGS + qnorm(0.975) * prop_se
  theta <- coef_BFGS[1]
  se_th <- prop_se[1]
  lo_th <- lo[1]
  up_th <- up[1]

  #### TRIAL LEVEL ####
  surv_eff <- subset(memo, select = c(center, surv_est))
  colnames(surv_eff)[2] <- "effect"
  surv_eff$endp <- "MAIN"

  pfs_eff <- subset(memo, select = c(center, resp_est))
  colnames(pfs_eff)[2] <- "effect"
  pfs_eff$endp <- "SURR"

  shihco <- rbind(surv_eff, pfs_eff)
  shihco$center <- as.numeric(shihco$center)

  shihco <- shihco[order(shihco$center, shihco$endp), ]
  shihco$endp <- as.factor(shihco$endp)
  shihco$effect <- as.numeric(shihco$effect)

  invisible(capture.output(model <- nlme::gls(effect ~ -1 + factor(endp), data = shihco,
                                        correlation = nlme::corCompSymm(form = ~ 1 | center),
                                        weights = nlme::varIdent(form = ~ 1 | endp),
                                        method = "ML",
                                        control = nlme::glsControl(maxIter = 25, msVerbose = TRUE))))

  rsquared <- intervals(model, which = "var-cov")$corStruct^2
  R2 <- as.vector(rsquared)[2]
  lo_R2 <- as.vector(rsquared)[1]
  up_R2 <- as.vector(rsquared)[3]

  Trial.R2 <- data.frame(cbind(R2, lo_R2, up_R2), stringsAsFactors = TRUE)
  colnames(Trial.R2) <- c("R2 Trial", "CI lower limit",
                          "CI upper limit")
  rownames(Trial.R2) <- c(" ")

  Indiv.GlobalOdds <- data.frame(cbind(theta, lo_th, up_th), stringsAsFactors = TRUE)
  colnames(Indiv.GlobalOdds) <- c("Global Odds", "CI lower limit",
                                  "CI upper limit")
  rownames(Indiv.GlobalOdds) <- c(" ")

  EstTreatEffects <- memo
  rownames(EstTreatEffects) <- NULL
  colnames(EstTreatEffects) <- c("trial", "surv_est", "surv_se", "resp_est", "resp_se", "cova", "sample_size")

  output_list <- list(Indiv.GlobalOdds = Indiv.GlobalOdds,
                      Trial.R2 = Trial.R2, EstTreatEffects=EstTreatEffects, Call = match.call())

  class(output_list) <- "survbin"

  return(output_list)

}

#' Provides a summary of the surrogacy measures for an object fitted with the 'survbin()' function.
#'
#' @method summary survbin
#'
#' @param object An object of class 'survbin' fitted with the 'survbin()' function.
#' @param ... ...
#'
#' @return The surrogacy measures with their 95% confidence intervals.
#' @export
#'
#' @examples
#' \dontrun{
#' data("colorectal")
#' fit_bin <- survbin(data = colorectal, true = surv, trueind = SURVIND, surrog = responder,
#'                    trt = TREAT, center = CENTER, trial = TRIAL, patientid = patientid)
#' summary(fit_bin)
#' }

summary.survbin <- function(object,...){
  cat("Surrogacy measures with 95% confidence interval \n\n")
  cat("Individual level surrogacy: ", "\n\n")
  cat("Global Odds: ", sprintf("%.4f", object$Indiv.GlobalOdds[1,1]), "[", sprintf("%.4f", object$Indiv.GlobalOdds[1,2]),";", sprintf("%.4f", object$Indiv.GlobalOdds[1,3]) , "]", "\n\n")
  cat("Trial level surrogacy: ", "\n\n")
  cat("R Square: ", sprintf("%.4f", object$Trial.R2[1,1]),"[", sprintf("%.4f", object$Trial.R2[1,2]),";", sprintf("%.4f", object$Trial.R2[1,3]) , "]", "\n\n")
}

#' Prints all the elements of an object fitted with the 'survbin()' function.
#'
#' @method print survbin
#'
#' @param x An object of class 'survbin' fitted with the 'survbin()' function.
#' @param ... ...
#'
#' @return The surrogacy measures with their 95% confidence intervals and the estimated treament effect on the surrogate and true endpoint.
#' @export
#'
#' @examples
#' \dontrun{
#' data("colorectal")
#' fit_bin <- survbin(data = colorectal, true = surv, trueind = SURVIND, surrog = responder,
#'                    trt = TREAT, center = CENTER, trial = TRIAL, patientid = patientid)
#' print(fit_bin)
#' }
print.survbin <- function(x,...){
  cat("Surrogacy measures with 95% confidence interval \n\n")
  cat("Individual level surrogacy: ", "\n\n")
  cat("Global Odds: ", sprintf("%.4f", x$Indiv.GlobalOdds[1,1]), "[", sprintf("%.4f", x$Indiv.GlobalOdds[1,2]),";", sprintf("%.4f", x$Indiv.GlobalOdds[1,3]) , "]", "\n\n")
  cat("Trial level surrogacy: ", "\n\n")
  cat("R Square: ", sprintf("%.4f", x$Trial.R2[1,1]),"[", sprintf("%.4f", x$Trial.R2[1,2]),";", sprintf("%.4f", x$Trial.R2[1,3]) , "]", "\n\n")

  cat("Estimated treatment effects on surrogate (resp_est) and survival (surv_est) endpoint: \n\n")
  print(x$EstTreatEffects[,c(1,2,3,4,5,7)], row.names = FALSE)
}

#' Generates a plot of the estimated treatment effects for the surrogate endpoint versus the estimated treatment effects for the true endpoint for an object fitted with the 'survbin()' function.
#'
#' @method plot survbin
#'
#' @param x An object of class 'survbin' fitted with the 'survbin()' function.
#' @param ... ...
#'
#' @return A plot of the type ggplot
#' @export
#'
#' @examples
#' \dontrun{
#' data("colorectal")
#' fit_bin <- survbin(data = colorectal, true = surv, trueind = SURVIND, surrog = responder,
#'                    trt = TREAT, center = CENTER, trial = TRIAL, patientid = patientid)
#' plot(fit_bin)
#' }
#'
plot.survbin <- function(x,...){
  if (requireNamespace("ggplot2", quietly = TRUE)) {
    resp_est <- surv_est <- sample_size <- NULL
    estimated_treatment_effects <- x$EstTreatEffects

    estimated_treatment_effects$sample_size <- as.numeric(estimated_treatment_effects$sample_size)
    estimated_treatment_effects$surv_est <- as.numeric(estimated_treatment_effects$surv_est)
    estimated_treatment_effects$resp_est <- as.numeric(estimated_treatment_effects$resp_est)

    # Create the scatter plot
    ggplot2::ggplot(data = estimated_treatment_effects, ggplot2::aes(x = resp_est, y = surv_est, size = sample_size)) +
      ggplot2::geom_point() +
      ggplot2::geom_smooth(method = "lm", se = FALSE, color = "royalblue3") +
      ggplot2::labs(x = "Treatment effect on surrogate", y = "Treatment effect on true") +
      ggplot2::ggtitle("Treatment effect on true endpoint vs. treatment effect on surrogate endpoint") +
      ggplot2::theme(legend.position="none")

  } else {
    stop("ggplot2 is not installed. Please install ggplot2 to use this function.")
  }
}


