% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dnnModel.R
\name{trainDeconvModel}
\alias{trainDeconvModel}
\title{Train deconvolution model for spatial transcriptomics data}
\usage{
trainDeconvModel(
  object,
  type.data.train = "mixed",
  type.data.test = "mixed",
  batch.size = 64,
  num.epochs = 60,
  num.hidden.layers = 2,
  num.units = c(200, 200),
  activation.fun = "relu",
  dropout.rate = 0.25,
  loss = "kullback_leibler_divergence",
  metrics = c("accuracy", "mean_absolute_error", "categorical_accuracy"),
  normalize = TRUE,
  scaling = "standardize",
  norm.batch.layers = TRUE,
  custom.model = NULL,
  shuffle = TRUE,
  sc.downsampling = NULL,
  use.generator = FALSE,
  on.the.fly = FALSE,
  agg.function = "AddRawCount",
  threads = 1,
  view.metrics.plot = TRUE,
  verbose = TRUE
)
}
\arguments{
\item{object}{\code{\linkS4class{SpatialDDLS}} object with
\code{single.cell.real}/\code{single.cell.simul}, \code{prob.cell.types},
and \code{mixed.profiles} slots (the last only if \code{on.the.fly =
  FALSE}).}

\item{type.data.train}{Type of profiles to be used for training. It can be
\code{'both'}, \code{'single-cell'} or \code{'mixed'} (\code{'mixed'} by
default).}

\item{type.data.test}{Type of profiles to be used for evaluation. It can be
\code{'both'}, \code{'single-cell'} or \code{'mixed'} (\code{'mixed'} by
default).}

\item{batch.size}{Number of samples per gradient update (64 by default).}

\item{num.epochs}{Number of epochs to train the model (60 by default).}

\item{num.hidden.layers}{Number of hidden layers of the neural network (2 by
default). This number must be equal to the length of \code{num.units}
argument.}

\item{num.units}{Vector indicating the number of neurons per hidden layer
(\code{c(200, 200)} by default). The length of this vector must be equal to
the \code{num.hidden.layers} argument.}

\item{activation.fun}{Activation function (\code{'relu'} by default). See
the
\href{https://tensorflow.rstudio.com/reference/keras/activation_relu.html}{keras
documentation} to know available activation functions.}

\item{dropout.rate}{Float between 0 and 1 indicating the fraction of
input neurons to be dropped in layer dropouts (0.25 by default). By
default, \pkg{SpatialDDLS} implements 1 dropout layer per hidden layer.}

\item{loss}{Character indicating loss function selected for model training
(\code{'kullback_leibler_divergence'} by default). See the
\href{https://tensorflow.rstudio.com/reference/keras/loss-functions.html}{keras
documentation} to know available loss functions.}

\item{metrics}{Vector of metrics used to assess model performance during
training and evaluation (\code{c("accuracy", "mean_absolute_error",
  "categorical_accuracy")} by default). See the
\href{https://tensorflow.rstudio.com/reference/keras/metric_binary_accuracy.html}{keras
documentation} to know available performance metrics.}

\item{normalize}{Whether to normalize data using logCPM (\code{TRUE} by
default). This parameter is only considered when the method used to
simulate mixed transcriptional profiles (\code{simMixedProfiles}
function) was \code{"AddRawCount"}. Otherwise, data were already
normalized.}

\item{scaling}{How to scale data before training. It can be:
\code{"standardize"} (values are centered around the mean with a unit
standard deviation), \code{"rescale"} (values are shifted and rescaled so
that they end up ranging between 0 and 1) or \code{"none"} (no
scaling is performed). \code{"standardize"} by default.}

\item{norm.batch.layers}{Whether to include batch normalization layers
between each hidden dense layer (\code{TRUE} by default).}

\item{custom.model}{It allows to use a custom neural network architecture. It
must be a \code{keras.engine.sequential.Sequential} object in which the
number of input neurons is equal to the number of considered
features/genes, and the number of output neurons is equal to the number of
cell types considered (\code{NULL} by default). If provided, the arguments
related to the neural network architecture will be ignored.}

\item{shuffle}{Boolean indicating whether data will be shuffled (\code{TRUE}
by default).}

\item{sc.downsampling}{It is only used if \code{type.data.train} is equal to
\code{'both'} or \code{'single-cell'}. It allows to set a maximum number of
single-cell profiles of a specific cell type for training to avoid
an unbalanced representation of classes (\code{NULL} by default).}

\item{use.generator}{Boolean indicating whether to use generators during
training and test. Generators are automatically used when \code{on.the.fly
  = TRUE} or HDF5 files are used, but it can be activated by the user on
demand (\code{FALSE} by default).}

\item{on.the.fly}{Boolean indicating whether simulated data will be generated
'on the fly' during training (\code{FALSE} by default).}

\item{agg.function}{If \code{on.the.fly == TRUE}, function used to build
mixed transcriptional profiles. It may be:
\itemize{ \item \code{"AddRawCount"} (by default): single-cell
profiles (raw counts) are added up across cells. Then, log-CPMs are
calculated. \item \code{"MeanCPM"}: single-cell profiles (raw counts) are
transformed into logCPM and cross-cell averages are calculated.
\item \code{"AddCPM"}: single-cell profiles (raw counts) are transformed
into CPMs and are added up across cells. Then, log-CPMs are calculated.}}

\item{threads}{Number of threads used during simulation of mixed
transcriptional profiles if \code{on.the.fly = TRUE} (1 by default).}

\item{view.metrics.plot}{Boolean indicating whether to show plots of loss and
evaluation metrics during training (\code{TRUE} by default). \pkg{keras}
for R allows to see model progression during training if you are working in
RStudio.}

\item{verbose}{Boolean indicating whether to display model progression during
training and model architecture information (\code{TRUE} by default).}
}
\value{
A \code{\linkS4class{SpatialDDLS}} object with \code{trained.model}
slot containing a \code{\linkS4class{DeconvDLModel}} object. For more
information about the structure of this class, see
\code{?\linkS4class{DeconvDLModel}}.
}
\description{
Train a deep neural network model using training data from the
\code{\linkS4class{SpatialDDLS}} object. This model will be used to
deconvolute spatial transcriptomics data from the same biological context as
the single-cell RNA-seq data used to train it. In addition, the trained
model is evaluated using test data, and prediction results are obtained to
determine its performance (see \code{?\link{calculateEvalMetrics}}).
}
\details{
\strong{Simulation of mixed transcriptional profiles 'on the fly'}

\code{trainDeconvModel} can avoid storing simulated mixed spot profiles by
using the \code{on.the.fly} argument. This functionality aims at reducing the
the \code{simMixedProfiles} function's memory usage: simulated profiles are
built in each batch during training/evaluation.

\strong{Neural network architecture}

It is possible to change the model's architecture: number of hidden layers,
number of neurons for each hidden layer, dropout rate, activation function,
and loss function. For more customized models, it is possible to provide a
pre-built model through the \code{custom.model} argument (a
\code{keras.engine.sequential.Sequential} object) where it is necessary that
the number of input neurons is equal to the number of considered
features/genes, and the number of output neurons is equal to the number of
considered cell types.
}
\examples{
\donttest{
set.seed(123)
sce <- SingleCellExperiment::SingleCellExperiment(
  assays = list(
    counts = matrix(
      rpois(30, lambda = 5), nrow = 15, ncol = 10,
      dimnames = list(paste0("Gene", seq(15)), paste0("RHC", seq(10)))
    )
  ),
  colData = data.frame(
    Cell_ID = paste0("RHC", seq(10)),
    Cell_Type = sample(x = paste0("CellType", seq(2)), size = 10,
                       replace = TRUE)
  ),
  rowData = data.frame(
    Gene_ID = paste0("Gene", seq(15))
  )
)
SDDLS <- createSpatialDDLSobject(
  sc.data = sce,
  sc.cell.ID.column = "Cell_ID",
  sc.gene.ID.column = "Gene_ID",
  sc.filt.genes.cluster = FALSE
)
SDDLS <- genMixedCellProp(
  object = SDDLS,
  cell.ID.column = "Cell_ID",
  cell.type.column = "Cell_Type",
  num.sim.spots = 50,
  train.freq.cells = 2/3,
  train.freq.spots = 2/3,
  verbose = TRUE
)
SDDLS <- simMixedProfiles(SDDLS)
SDDLS <- trainDeconvModel(
  object = SDDLS,
  batch.size = 12,
  num.epochs = 5
)
}
  
}
\seealso{
\code{\link{plotTrainingHistory}} \code{\link{deconvSpatialDDLS}}
}
