#' Annotate Seurat Object with SlimR Cell Type Predictions
#'
#' @description This function assigns SlimR predicted cell types to a Seurat object
#'     based on cluster annotations, and stores the results in the meta.data slot.
#'
#' @param seurat_obj A Seurat object containing cluster information in meta.data.
#' @param cluster_col Character string indicating the column name in meta.data that
#'     contains cluster IDs.
#' @param SlimR_anno_result List generated by function Celltype_Calculate() which
#'     containing a data.frame in $Prediction_results with: 1.cluster_col (Cluster
#'     identifiers (should match cluster_col in meta.data)) 2.Predicted_cell_type
#'     (Predicted cell types for each cluster).
#' @param plot_UMAP logical(1); if TRUE, plot the UMAP with cell type annotations.
#' @param annotation_col The location to write in 'meta.data' that contains the
#'     predicted cell type. (default = "Cell_type_SlimR")
#'
#' @return A Seurat object with updated meta.data containing the predicted cell types.
#' @note If plot_UMAP = TRUE, this function will print a UMAP plot as a side effect.
#'
#' @export
#' @family Automated_Annotation_Workflow
#'
#' @importFrom Seurat RenameIdents Idents
#' @importFrom dplyr as_tibble
#'
#' @examples
#' \dontrun{
#' sce <- Celltype_Annotation(seurat_obj = sce,
#'     cluster_col = "seurat_clusters",
#'     SlimR_anno_result = SlimR_anno_result,
#'     plot_UMAP = TRUE,
#'     annotation_col = "Cell_type_SlimR"
#'     )
#'     }
#'
Celltype_Annotation<- function(
    seurat_obj,
    cluster_col,
    SlimR_anno_result,
    plot_UMAP = TRUE,
    annotation_col = "Cell_type_SlimR"
) {
  if (!inherits(seurat_obj, "Seurat")) {
    stop("Input must be a Seurat object")
  }

  if (!(cluster_col %in% colnames(seurat_obj@meta.data))) {
    stop(sprintf("Cluster column '%s' not found in meta.data", cluster_col))
  }

  if (!is.list(SlimR_anno_result) || !("Prediction_results" %in% names(SlimR_anno_result))) {
    stop("SlimR_anno_result must be a list containing 'Prediction_results' data.frame")
  }

  prediction_df <- SlimR_anno_result[["Prediction_results"]]

  if (!is.data.frame(prediction_df)) {
    stop("'Prediction_results' must be a data.frame")
  }

  required_columns <- c("cluster_col", "Predicted_cell_type")
  if (!all(required_columns %in% colnames(prediction_df))) {
    missing <- setdiff(required_columns, colnames(prediction_df))
    stop(sprintf("Missing required columns in Prediction_results: %s", paste(missing, collapse = ", ")))
  }

  obj_clusters <- as.character(unique(seurat_obj@meta.data[[cluster_col]]))
  pred_clusters <- as.character(unique(prediction_df$cluster_col))

  if (length(obj_clusters) != length(pred_clusters)) {
    stop("Cluster count mismatch: ",
         "Seurat object has ", length(obj_clusters), " clusters, \n",
         "  SlimR_anno_result has ", length(pred_clusters), " clusters. \n",
         "  Please check your input cluster_col.")
  }

  prediction_df <- dplyr::as_tibble(prediction_df)

  mapping <- setNames(
    as.character(prediction_df$Predicted_cell_type),
    as.character(prediction_df$cluster_col)
  )

  obj_clusters <- as.character(seurat_obj@meta.data[[cluster_col]])
  pred_clusters <- as.character(prediction_df$cluster_col)

  missing_clusters <- setdiff(
    as.character(pred_clusters),
    as.character(obj_clusters)
  )

  if (length(missing_clusters) > 0) {
    warning(sprintf("The following clusters in prediction results do not exist in the object: %s",
                    paste(missing_clusters, collapse = ", ")))
  }

  Seurat::Idents(seurat_obj) <- seurat_obj@meta.data[[cluster_col]]
  seurat_obj <- Seurat::RenameIdents(seurat_obj, mapping)

  seurat_obj@meta.data[[annotation_col]] <- Seurat::Idents(seurat_obj)

  p <- Seurat::DimPlot(seurat_obj, reduction = "umap", label = TRUE, pt.size = 0.5) + Seurat::NoLegend() + Seurat::NoAxes()

  if (plot_UMAP) {
    print(p)
  }

  message(paste0("SlimR annotation: The cell types information predicted by SlimR in 'SlimR_anno_result$Prediction_results$Predicted_cell_type' has been written into 'seurat_obj@meta.data$",annotation_col,"."))

  return(seurat_obj)
}
