

#' Model matrix from hierarchies and/or a formula
#' 
#' A common interface to \code{\link{Hierarchies2ModelMatrix}}, \code{\link{Formula2ModelMatrix}} and \code{\link{HierarchiesAndFormula2ModelMatrix}}
#' 
#' @param data Matrix or data frame with data containing codes of relevant variables
#' @param hierarchies List of hierarchies, which can be converted by \code{\link{AutoHierarchies}}.
#' Thus, the variables can also be coded by \code{"rowFactor"} or \code{""}, which correspond to using the categories in the data.
#' @param formula A model formula
#' @param inputInOutput Logical vector (possibly recycled) for each element of hierarchies.
#'         TRUE means that codes from input are included in output. Values corresponding to \code{"rowFactor"} or \code{""} are ignored.
#' @param crossTable Cross table in output when TRUE
#' @param sparse Sparse matrix in output when TRUE (default)
#' @param viaOrdinary When TRUE, output is generated by \code{\link{model.matrix}} or \code{\link{sparse.model.matrix}}.
#'                    Since these functions omit a factor level, an empty factor level is first added. 
#' @param total String used to name totals 
#' @param removeEmpty When TRUE, empty columns (only zeros) are not included in output (relevant when hierarchies)    
#' @param ... Further arguments to  \code{\link{Hierarchies2ModelMatrix}}, \code{\link{Formula2ModelMatrix}} or \code{\link{HierarchiesAndFormula2ModelMatrix}}    
#'
#' @return A sparse model matrix or a list of two elements (model matrix and cross table)
#' @export
#' @author Øyvind Langsrud
#' 
#' @examples
#' # Create some input
#' z <- SSBtoolsData("sprt_emp_withEU")
#' z$age[z$age == "Y15-29"] <- "young"
#' z$age[z$age == "Y30-64"] <- "old"
#' ageHier <- data.frame(mapsFrom = c("young", "old"), mapsTo = "Total", sign = 1)
#' geoDimList <- FindDimLists(z[, c("geo", "eu")], total = "Europe")[[1]]
#' 
#' # Small dataset example. Two dimensions.
#' s <- z[z$geo == "Spain" & z$year != 2016, ]
#' 
#' # via Hierarchies2ModelMatrix() and converted to ordinary matrix (not sparse)
#' ModelMatrix(s, list(age = ageHier, year = ""), sparse = FALSE)
#' 
#' # Hierarchies generated automatically. Then via Hierarchies2ModelMatrix()
#' ModelMatrix(s[, c(1, 3)])
#' 
#' # via Formula2ModelMatrix()
#' ModelMatrix(s, formula = ~age + year)
#' 
#' # via model.matrix() after adding empty factor levels
#' ModelMatrix(s, formula = ~age + year, sparse = FALSE, viaOrdinary = TRUE)
#' 
#' # via sparse.model.matrix() after adding empty factor levels
#' ModelMatrix(s, formula = ~age + year, viaOrdinary = TRUE)
#' 
#' # via HierarchiesAndFormula2ModelMatrix() and using different data and parameter settings
#' ModelMatrix(s, list(age = ageHier, geo = geoDimList, year = ""), formula = ~age * geo + year, 
#'             inputInOutput = FALSE, removeEmpty = TRUE, crossTable = TRUE)
#' ModelMatrix(s, list(age = ageHier, geo = geoDimList, year = ""), formula = ~age * geo + year, 
#'             inputInOutput = c(TRUE, FALSE), removeEmpty = FALSE, crossTable = TRUE)
#' ModelMatrix(z, list(age = ageHier, geo = geoDimList, year = ""), formula = ~age * year + geo, 
#'             inputInOutput = c(FALSE, TRUE), crossTable = TRUE)
ModelMatrix <- function(data, hierarchies = NULL, formula = NULL,
                        inputInOutput = TRUE,
                        crossTable = FALSE, sparse = TRUE,
                        viaOrdinary = FALSE,
                        total = "Total",
                        removeEmpty = FALSE,
                           ...) {
  
  if (is.null(formula) & is.null(hierarchies)) {
    hierarchies <- FindHierarchies(data, total = total)
  }
  
  if(viaOrdinary){
    if(!is.null(hierarchies) | crossTable){
      warning("viaOrdinary ignorded")
    } else {
      return(Model_Matrix(formula = formula, data = data, sparse = sparse)) 
    }
  }
  
  if (!is.null(formula) & !is.null(hierarchies)) {
    a <- HierarchiesAndFormula2ModelMatrix(data = data, hierarchies = hierarchies, formula = formula, inputInOutput = inputInOutput, crossTable = crossTable, total = total, removeEmpty = removeEmpty, ...)
  }
  
  if (is.null(formula) & !is.null(hierarchies)) {
    a <- Hierarchies2ModelMatrix(data = data, hierarchies = hierarchies, inputInOutput = inputInOutput, crossTable = crossTable, total = total, removeEmpty = removeEmpty,  ...)
  }
  
  if (!is.null(formula) & is.null(hierarchies)) {
    a <- Formula2ModelMatrix(data = data, formula = formula, crossTable = crossTable, total=total, ...)
  }
  
  if (crossTable) {
    if (!is.data.frame(a$crossTable)) {
      a$crossTable <- as.data.frame(a$crossTable)
    }
  }
  
  if(!sparse){
    if(crossTable){
      a$modelMatrix <- as.matrix(a$modelMatrix)
    } else {
      a <- as.matrix(a)
    }
  }
  a
}
  


#' Overparameterized model matrix
#'
#' All factor levels included
#'
#' @param formula formula
#' @param data data frame
#' @param mf model frame (alternative input instead of data)
#' @param allFactor When TRUE all variables are coerced to factor
#' @param sparse When TRUE sparse matrix created by sparse.model.matrix()
#'
#' @return model matrix created via model.matrix() or sparse.model.matrix()
#' @importFrom stats model.frame model.matrix
#' @importFrom Matrix sparse.model.matrix
#' @export
#' @keywords internal
#'
#' @examples
#'   z <- SSBtoolsData("sprt_emp_withEU")
#'   Model_Matrix(~age*year + geo, z)
Model_Matrix <- function(formula, data = NULL, mf = model.frame(formula, data = data), allFactor = TRUE, sparse = FALSE)  {
  
  for (i in 1:length(mf)) {
    if (allFactor)
      mf[[i]] <- as.factor(mf[[i]])
    if (is.factor(mf[[i]]))
      mf[[i]] <- AddEmptyLevel(mf[[i]])
  }
  if (sparse)
    return(sparse.model.matrix(formula, data = mf))
  model.matrix(formula, data = mf)
}


AddEmptyLevel <- function(x) factor(x, levels = c("tu1lnul1", levels(x)))

