% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/buildKrigingForrester.R
\name{buildKriging}
\alias{buildKriging}
\title{Build Kriging Model}
\usage{
buildKriging(x, y, control = list())
}
\arguments{
\item{x}{design matrix (sample locations)}

\item{y}{vector of observations at \code{x}}

\item{control}{(list), with the options for the model building procedure. Note: This can also be changed after the model has been built, by manipulating the respective \code{object$target} value.
\describe{
    \item{\code{types}}{a character vector giving the data type of each variable.
    All but "factor" will be handled as numeric, "factor" (categorical) variables
    will be subject to the hamming distance.}
    \item{\code{thetaLower}}{lower boundary for theta, default is \code{1e-4}}
    \item{\code{thetaUpper}}{upper boundary for theta, default is \code{1e2}}
    \item{\code{algTheta}}{algorithm used to find theta, default is \code{optimDE}.}
    \item{\code{budgetAlgTheta}}{budget for the above mentioned algorithm,
    default is \code{200}. The value will be multiplied with the length of
    the model parameter vector to be optimized.}
    \item{\code{optimizeP}}{boolean that specifies whether the exponents (\code{p}) should be optimized. Else they will be set to two. Default is \code{FALSE}.}
    \item{\code{useLambda}}{whether or not to use the regularization constant lambda
    (nugget effect). Default is \code{TRUE}.}
    \item{\code{lambdaLower}}{lower boundary for log10{lambda}, default is \code{-6}}
    \item{\code{lambdaUpper}}{upper boundary for log10{lambda}, default is \code{0}}
    \item{\code{startTheta}}{optional start value for theta optimization, default is \code{NULL}}
    \item{\code{reinterpolate}}{whether (\code{TRUE},default) or not (\code{FALSE}) reinterpolation
    should be performed.}
    \item{\code{target}}{values of the prediction, a vector of strings.
    Each string specifies a value to be predicted, e.g., "y" for mean, "s" for standard deviation, "ei" for expected improvement.
    See also \code{\link{predict.kriging}}.}
}}
}
\value{
an object of class \code{kriging}. Basically a list, with the options
and found parameters for the model which has to be passed to the predictor function:\cr
\code{x} sample locations (scaled to values between 0 and 1)\cr
\code{y} observations at sample locations (see parameters)\cr
\code{thetaLower} lower boundary for theta (see parameters)\cr
\code{thetaUpper} upper boundary for theta (see parameters)\cr
\code{algTheta} algorithm to find theta (see parameters)\cr
\code{budgetAlgTheta} budget for the above mentioned algorithm (see parameters)\cr
\code{optimizeP} boolean that specifies whether the exponents (\code{p}) were optimized (see parameters)\cr
\code{normalizeymin} minimum in normalized space\cr
\code{normalizeymax} maximum in normalized space\cr
\code{normalizexmin} minimum in input space\cr
\code{normalizexmax} maximum in input space\cr
\code{dmodeltheta} vector of activity parameters\cr
\code{Theta} log_10 vector of activity parameters (i.e. \code{log10(dmodeltheta)})\cr
\code{dmodellambda} regularization constant (nugget) \cr
\code{Lambda} log_10 of regularization constant (nugget) (i.e. \code{log10(dmodellambda)})\cr
\code{yonemu} \code{Ay-ones*mu} \cr
\code{ssq} sigma square\cr
\code{mu} mean mu\cr
\code{Psi} matrix large Psi\cr
\code{Psinv} inverse of Psi\cr
\code{nevals} number of Likelihood evaluations during MLE
}
\description{
This function builds a Kriging model based on code by Forrester et al..
By default exponents (p) are fixed at a value of two, and a nugget (or regularization constant) is used.
To correct the uncertainty estimates in case of nugget, re-interpolation is also by default turned on.
}
\details{
The model uses a Gaussian kernel: \code{k(x,z)=exp(-sum(theta_i * |x_i-z_i|^p_i))}. By default, \code{p_i = 2}.
Note that if dimension \code{x_i} is a factor variable (see parameter \code{types}), Hamming distance will be used
instead of \code{|x_i-z_i|}.
}
\examples{
## Create design points
set.seed(1)
x <- cbind(runif(20)*15-5,runif(20)*15)
y <- funBranin(x)
## Create model with default settings
fit <- buildKriging(x,y,control = list(algTheta=optimLHD))
## Print model parameters
print(fit)
## Predict at new location
predict(fit,cbind(1,2))
## True value at location
funBranin(matrix(c(1,2), 1))
##
## Next Example: Handling factor variables
\donttest{
## create a test function:
braninFunctionFactor <- function (x) {
y <- (x[2]  - 5.1/(4 * pi^2) * (x[1] ^2) + 5/pi * x[1]  - 6)^2 +
		10 * (1 - 1/(8 * pi)) * cos(x[1] ) + 10
if(x[3]==1)
		y <- y +1
else if(x[3]==2)
		y <- y -1
y
}
## create training data
set.seed(1)
x <- cbind(runif(50)*15-5,runif(50)*15,sample(1:3,50,replace=TRUE))
y <- as.matrix(apply(x,1,braninFunctionFactor))
## fit the model (default: assume all variables are numeric)
fitDefault <- buildKriging(x,y,control = list(algTheta=optimDE))
## fit the model (give information about the factor variable)
fitFactor <- buildKriging(x,y,control =
	list(algTheta=optimDE,types=c("numeric","numeric","factor")))
## create test data
xtest <- cbind(runif(200)*15-5,runif(200)*15,sample(1:3,200,replace=TRUE))
ytest <- as.matrix(apply(xtest,1,braninFunctionFactor))
## Predict test data with both models, and compute error
ypredDef <- predict(fitDefault,xtest)$y
ypredFact <- predict(fitFactor,xtest)$y
mean((ypredDef-ytest)^2)
mean((ypredFact-ytest)^2)
}
}
\references{
Forrester, Alexander I.J.; Sobester, Andras; Keane, Andy J. (2008). Engineering Design via Surrogate Modelling - A Practical Guide. John Wiley & Sons.
}
\seealso{
\code{\link{predict.kriging}}
}
