% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SLFPCA.R
\name{SLFPCA}
\alias{SLFPCA}
\title{Sparse logistic functional principal component analysis}
\usage{
SLFPCA(
  Ly,
  Lt,
  interval,
  npc,
  nknots,
  norder,
  kappa_theta,
  sparse_pen,
  nRegGrid = 51,
  bwmu_init = 0.5,
  bwcov_init = 1,
  stepmu,
  mucand_num = 100,
  itermax = 100,
  tol = 0.5
)
}
\arguments{
\item{Ly}{A \code{list} of \emph{n} vectors, where \emph{n} is the sample size. Each entry contains the binary measurements of each subject at the observation time correspond to \code{Lt}.}

\item{Lt}{A \code{list} of \emph{n} vectors, where \emph{n} is the sample size. Each entry contains the observation time in ascending order for each subject.}

\item{interval}{A \code{vector} of length two denoting the supporting interval.}

\item{npc}{An integer denoting the number of FPCs.}

\item{nknots}{An integer denoting the number of interior knots for the using B-spline basis.}

\item{norder}{An integer denoting the order of the using B-spline basis, which is one higher than their degree.}

\item{kappa_theta}{A \code{vector} denoting the smoothing parameters for eigenfunctions, the optimal tuning parameter is chosen from them.}

\item{sparse_pen}{A \code{vector} denoting the sparseness parameters for eigenfunctions, the optimal tuning parameter is chosen from them.}

\item{nRegGrid}{An integer denoting the number of equally spaced time points in the supporting interval. The eigenfunctions and mean function are estimated at these equally spaced time points first, before transforming into functional data object. (default: 51)}

\item{bwmu_init}{A scalar denoting the bandwidth for mean function estimation in the setting of initial values. (default: 0.5)}

\item{bwcov_init}{A scalar denoting the bandwidth for covariance function estimation in the setting of initial values. (default: 1)}

\item{stepmu}{A scalar denoting the length between each considered smoothing parameter for mean function. For selection of smoothing parameter for mean function, we start from zero and increase the value until GCV score begins increasing.}

\item{mucand_num}{An integer denoting the maximum number of the considered smoothing parameter for mean function. (default: 100)}

\item{itermax}{An integer denoting the maximum number of iterations. (default: 100)}

\item{tol}{A scalar. When difference of the loglikelihood functions between two consecutive iteration is less than \code{tol}, the convergence is supposed to be reached. (default: 0.5)}
}
\value{
A \code{list} containing the following components:
\item{mufd}{A functional data object for the mean function estimate.}
\item{eigfd_list}{A \code{list} containing \code{npc} functional data objects, which are the eigenfunction estimates.}
\item{score}{A \emph{n} by \code{npc} \code{matrix} containing the estimates of the FPC scores, where \emph{n} is the sample size.}
\item{kappa_mu}{A scalar denoting the selected smoothing parameter for mean function.}
\item{kappa_theta}{A scalar denoting the selected smoothing parameter for eigenfunctions.}
\item{sparse_pen}{A scalar denoting the selected sparseness parameter for eigenfunctions.}
}
\description{
Sparse logistic functional principal component analysis (SLFPCA) for binary data. The estimated eigenfunctions from SLFPCA can be strictly zero on some sub-intervals, which is helpful for interpretation.
}
\examples{
#Generate data
n <- 100
npc <- 1
interval <- c(0, 10)
gridequal <- seq(0, 10, length.out = 51)
basis <- fda::create.bspline.basis(c(0, 10), nbasis = 13, norder = 4,
         breaks = seq(0, 10, length.out = 11))
meanfun <- function(t){2 * sin(pi * t/5)/sqrt(5)}
lambda_1 <- 3^2 #the first eigenvalue
score <- cbind(rnorm(n, 0, sqrt(lambda_1)))
eigfun <- list()
eigfun[[1]] <- function(t){cos(pi * t/5)/sqrt(5)}
eigfd <- list()
for(i in 1:npc){
  eigfd[[i]] <- fda::smooth.basis(gridequal, eigfun[[i]](gridequal), basis)$fd
}
DataNew <- GenBinaryFD(n, interval, sparse = 8:12, regular = FALSE,
           meanfun = meanfun, score, eigfd)
SLFPCA_list <- SLFPCA(DataNew$Ly, DataNew$Lt, interval, npc, nknots = 9, norder = 4,
               kappa_theta = 0.2, sparse_pen = 0,
               nRegGrid = 51, stepmu = 0.005)
plot(SLFPCA_list$eigfd_list[[1]])

}
