% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/soundSat.R
\name{soundSat}
\alias{soundSat}
\title{Soundscape Saturation Index}
\usage{
soundSat(
  soundpath,
  channel = "stereo",
  timeBin = 60,
  dbThreshold = -90,
  targetSampRate = NULL,
  wl = 512,
  window = signal::hamming(wl),
  overlap = ceiling(length(window)/2),
  histbreaks = "FD",
  powthr = c(5, 20, 1),
  bgnthr = c(0.5, 0.9, 0.05),
  normality = "ad.test",
  backup = NULL
)
}
\arguments{
\item{soundpath}{single directory or multiple directory to audio files. The directory must lead to a single folder or a combination of folders.}

\item{channel}{channel where the saturation values will be extract from. Available channels are: \code{"stereo"}, \code{"mono"}, \code{"left"} or \code{"right"}. Defaults to \code{"stereo"}.}

\item{timeBin}{size (in seconds) of the time bin. Defaults to \code{60}.}

\item{dbThreshold}{minimum allowed value of dB for the spectrograms. Defaults to \code{-90}, as set by Towsey.}

\item{targetSampRate}{sample rate of the audios. Defaults to \code{NULL} to not change the sample rate. This argument is only used to down sample the audio.}

\item{wl}{window length of the spectrogram. Defaults to \code{512}.}

\item{window}{window used to smooth the spectrogram. Defaults to \code{signal::hammning(wl)}. Switch to \code{signal::hanning(wl)} if to use hanning instead.}

\item{overlap}{overlap between the spectrogram windows. Defaults to \code{wl/2} (half the window length)}

\item{histbreaks}{breaks used to calculate Background Noise. Available breaks are: \code{"FD"}, \code{"Sturges"}, \code{"scott"} and \code{100}. Defaults to \code{"FD"}.
\if{html}{\out{<br>}}Can also be set to any number to limit or increase the amount of breaks.}

\item{powthr}{vector of values to evaluate the activity matrix for Soundscape Power (in dB). The first value corresponds to the lowest dB value and the second corresponds to the highest, the third value is the step.
\if{html}{\out{<br>}}Defaults to \code{c(5, 20, 1)}, which means the first thresholds starts at 5dB and jumps a whole number till 20dB.}

\item{bgnthr}{vector of values to evaluate the activity matrix for Background Noise (in \%). The first value corresponds to the lowest quantile value and the second corresponds to the highest, the third value is the step.
\if{html}{\out{<br>}}Defaults to \code{c(0.5, 0.9, 0.05)}, which means the first thresholds starts at 50\% and jumps 5\% till 90\%.}

\item{normality}{normality test to determine which threshold combination has the most normal distribution of values. We recommend to pick any test from the \code{nortest} package. Input the test as a character. Defaults to \code{"ad.test"}.
\if{html}{\out{<br>}}\code{"ks.test"} is not available. \code{"shapiro.test"} can be used, however we recommend using only when analyzing very few recordings.}

\item{backup}{path to backup the saturation values in case the computer is turned off during processing or in case you cannot be sure the computer will be on for the entire process. Defaults to \code{NULL}.
\if{html}{\out{<br>}}The backup will be updated every 5 recordings processed.}
}
\value{
A list containing five objects. The first and second objects (powthresh and bgnthresh) are the threshold values that yielded the most normal distribution of saturation values using the normality test set by the user. The third (normality) contains the statitics values of the normality test that yielded the most normal distribution. The fourth object (values) contains a data.frame with the the values of saturation for each bin of each recording and the size of the bin in seconds. The fifth contains a data.frame with errors that occurred with specific files during the function.
}
\description{
Calculate Soundscape Saturation for a combination of recordings using Burivalova 2018 methodology
}
\details{
Soundscape Saturation (\code{SAT}) is a measure of the proportion of frequency bins that are acoustically active in a determined window of time. It was developed by Burivalova et al. 2017 as an index to test the acoustic niche hypothesis.
To calculate this function, first we need to generate an activity matrix for each time bin of your recording with the following formula:

\deqn{a_{mf} = 1\  if (BGN_{mf} > \theta_{1})\  or\  (POW_{mf} > \theta_{2});\  otherwise,\  a_{mf} = 0,}

Where \eqn{\theta_{1}} is the threshold of BGN values and \eqn{\theta_{2}} is a threshold of dB values.
Since we define a interval for both the threshold, this means that an activity matrix will be generated for each bin of each recording.
For each combination of threshold a SAT measure will be taken with the following formula:

\deqn{S_{m} = \frac{\sum_{f = 1}^N a_{mf}}{N}}

After these equations are done, we check every threshold combination for normality and pick the combination that yields the most normal distribution of saturation values.

If you set a path for the "path" argument, a single rds file will be written in your path. This objects can be loaded again through the "satBack" function to continue the calculation of saturation in case the process is suddenly stopped.
}
\examples{
\donttest{
### Downloading audiofiles from public Zenodo library
dir <- tempdir()
recName <- paste0("GAL24576_20250401_", sprintf("\%06d", seq(0, 200000, by = 50000)),".wav")
recDir <- paste(dir, recName, sep = "/")

for(rec in recName) {
 print(rec)
 url <- paste0("https://zenodo.org/records/17575795/files/", rec, "?download=1")
 download.file(url, destfile = paste(dir, rec, sep = "/"), mode = "wb")
}

### Running the function
sat <- soundSat(dir, wl = 256)

### Preparing the plot
timeSplit <- strsplit(sat$values$AUDIO, "_")
sides <- ifelse(grepl("left", sat$values$BIN), "left", "right")
date <- sapply(timeSplit, function(x)
  x[2])
time <- sapply(timeSplit, function(x)
  substr(x[3],1,6))
datePos <- paste(substr(date,1,4), substr(date,5,6), substr(date,7,8), sep = "-")
timePos <- paste(substr(time,1,2), substr(time,3,4), substr(time,5,6), sep = ":")
dateTime <- as.POSIXct(paste(datePos, timePos), format = "\%Y-\%m-\%d \%H:\%M:\%OS")
leftEar <- data.frame(SAT = sat$values$SAT[sides == "left"], HOUR = dateTime[sides == "left"])
rightEar <- data.frame(SAT = sat$values$SAT[sides == "right"], HOUR = dateTime[sides == "right"])

### Plotting results

plot(SAT~HOUR, data = leftEar, ylim = c(range(sat$values$SAT)),
col = "darkgreen", pch = 16,
     ylab = "Soundscape Saturation (\%)", xlab = "Time of Day", type = "b")
points(SAT~HOUR, data = rightEar, ylim = c(range(sat$values$SAT)),
col = "red", pch = 16, type = "b")
legend("topright", legend = c("Left Ear", "Right Ear"),
       col = c("darkgreen", "red"), lty = 1)

unlink(recDir)
}
}
\references{
Burivalova, Z., Towsey, M., Boucher, T., Truskinger, A., Apelis, C., Roe, P., & Game, E. T. (2017). Using soundscapes to detect variable degrees of human influence on tropical forests in Papua New Guinea. Conservation Biology, 32(1), 205-215. https://doi.org/10.1111/cobi.12968
}
