% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ICCstats.r
\name{ICCstats}
\alias{ICCstats}
\title{Intraclass correlation statistics from an lm.rrpp model fits}
\usage{
ICCstats(
  fit,
  subjects = NULL,
  with_in = NULL,
  groups = NULL,
  multivariate = FALSE,
  print.AOV = TRUE
)
}
\arguments{
\item{fit}{The \code{\link{lm.rrpp}}, previously evaluated.}

\item{subjects}{A single character value indicating which term in an ANOVA table
corresponds to research subjects.}

\item{with_in}{One or more character values indicating which terms in an ANOVA table
are measured within subjects (replications, plus maybe interactions).  If NULL,
the only replication within-subject will be considered as residuals.}

\item{groups}{An optional character value to indicate if a factor in the 
model frame of the \code{\link{lm.rrpp}} fit that could account for subject variation.
Using this argument might minimize the importance of subject variation, if subjects
have disparate values that could inflate ICC.  Note that this name could be different
than what is shown in the ANOVA table, if \code{\link{measurement.error}}
was used.  Use names(fit$LM$data), substituting fit with the name assigned
to the \code{\link{measurement.error}} object, to know the groups factor, 
if used.}

\item{multivariate}{Logical value for whether to include to calculate ICC matrix 
generalizations and perform eigenanalysis.}

\item{print.AOV}{Logical value for whether to include ANOVA table as screen 
output, when calculating ISS statistics.

Note that this function can return ICC statistics, even if they do not make sense.
It is possible to generate ICC stats with any ANOVA table, with at least one
term.}
}
\value{
Objects of class "ICCstats" return the following:
 \item{ICC_disp}{The intraclass correlation coefficient (ICC) based on the dispersion of values.}
 \item{ICC_mult}{The eigenvalues of ICC matrices}
}
\description{
Function performs analyses concerned with the repeatability (reliability) of multivariate data 
(measurements) collected from the same research subjects.  Although there is no
requirement for repeated measurements on all research subjects, the analysis assumes
that multiple observations are made.
}
\details{
Function uses ANOVA statistics or SSCP matrices to find the ratio of among-subject
to within-subject variance.  The former is a dispersion-based approach and the latter
is a multivariate generalization of the ICC statistic (as a matrix product).  The
multivariate generalizations of the statistics
described by Liljequist et al. (2019) are used to find matrix products, 
from which eigenanalysis is performed, providing ICC statistics by eigenvectors.  

Three statistics describe the ICC for the population,
agreement of measurements among subjects, and consistency between measurements.  
The last statistic does not necessarily measure the sameness 
between measurements but the consistency of change between measurements,
which might be indicative of a systematic measurement error.  
If groups are used, these three statistics are 
repeated, using the SSCP for groups-adjusted data.  
This approach accounts for group differences,
which would avoid large subject variation compared to measurement error 
inflating ICC values.  If there are 
inherently disparate groups from which subjects are sampled, 
this approach can elucidate better agreement and 
consistency in light of group differences.

This function is most useful for analyses performed with 
\code{\link{measurement.error}}, but any \code{\link{lm.rrpp}} fit can be used,
so long as research subjects can be defined. 

It is essential that all arguments are terms that can be found in the model frame
of the model fit, as provoke by ANOVA.  Using anova(fit) will elucidate the row
names of the ANOVA that could be used.
}
\examples{
\dontrun{
# Measurement error analysis on simulated data of fish shapes

data(fishy)

# Analysis unconcerned with groups 

ME1 <- measurement.error(
  Y = "coords",
  subjects = "subj",
  replicates = "reps",
  data = fishy)

anova(ME1)
ICCstats(ME1, subjects = "Subjects", with_in = "Systematic ME")

# Analysis concerned with groups 

ME2 <- measurement.error(
  Y = "coords",
  subjects = "subj",
  replicates = "reps",
  groups = "groups",
  data = fishy)

anova(ME2)
ICCstats(ME2, subjects = "Subjects", 
  with_in = "Systematic ME", groups = "groups")
ICCstats(ME2, subjects = "Subjects", 
  with_in = c("Systematic ME", "Systematic ME:Groups"), 
  groups = "groups")
  }
  
}
\references{
Liljequist, D., Elfving, B., & Skavberg Roaldsen, K. (2019). Intraclass correlation–A discussion 
and demonstration of basic features. PloS one, 14(7), e0219854.
}
\author{
Michael Collyer
}
\keyword{analysis}
