% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/f_latentIV.R
\name{latentIV}
\alias{latentIV}
\title{Fitting Linear Models with one Endogenous Regressor using Latent Instrumental Variables}
\usage{
latentIV(formula, data, start.params = c(), optimx.args = list(),
  verbose = TRUE)
}
\arguments{
\item{formula}{A symbolic description of the model to be fitted. Of class "formula".}

\item{data}{A data.frame containing the data of all parts specified in the formula parameter.}

\item{start.params}{A named vector containing a set of parameters to use in the first optimization iteration.
The names have to correspond exactly to the names of the components specified in the formula parameter.
If not provided, a linear model is fitted to derive them.}

\item{optimx.args}{A named list of arguments which are passed to \code{\link[optimx]{optimx}}. This allows users to tweak optimization settings to their liking.}

\item{verbose}{Show details about the running of the function.}
}
\value{
An object of classes \code{rendo.latent.IV} and \code{rendo.base} is returned which is a list and contains the following components:
\item{formula}{The formula given to specify the fitted model.}
\item{terms}{The terms object used for model fitting.}
\item{model}{The model.frame used for model fitting.}
\item{coefficients}{A named vector of all coefficients resulting from model fitting.}
\item{names.main.coefs}{a vector specifying which coefficients are from the model. For internal usage.}
\item{start.params}{A named vector with the initial set of parameters used to optimize the log-likelihood function.}
\item{res.optimx}{The result object returned by the function \code{optimx} after optimizing the log-likelihood function.}
\item{hessian}{A named, symmetric matrix giving an estimate of the Hessian at the found solution.}
\item{m.delta.diag}{A diagonal matrix needed when deriving the vcov to apply the delta method on theta5 which was transformed during the LL optimization.}
\item{fitted.values}{Fitted values at the found optimal solution.}
\item{residuals}{The residuals at the found optimal solution.}

The function \code{summary} can be used to obtain and print a summary of the results.
The generic accessor functions \code{coefficients}, \code{fitted.values}, \code{residuals}, \code{vcov}, \code{confint}, \code{logLik}, \code{AIC}, \code{BIC}, \code{case.names}, and \code{nobs} are available.
}
\description{
Fits linear models with one endogenous regressor and no additional explanatory variables using the latent instrumental variable approach
presented in Ebbes, P., Wedel, M.,  Böckenholt, U., and Steerneman, A. G. M. (2005). This is a statistical technique to address the endogeneity problem where no external instrumental
variables are needed. The important assumption of the model is that the latent variables are discrete with at least two groups with different means and
the structural error is normally distributed.
}
\details{
Let's consider the model:
\ifelse{html}{\out{<br><center>Y<sub>t</sub>=&beta;<sub>0</sub>+&alpha;P<sub>t</sub>+&epsilon;<sub>t</sub></center>}}{ \deqn{Y_{t} = \beta_{0} + \alpha P_{t} + \epsilon_{t}}}
\ifelse{html}{\out{<center>P<sub>t</sub>=&pi;'Z<sub>t</sub>+&nu;<sub>t</sub></center>}}{ \deqn{P_{t}=\pi^{'}Z_{t} + \nu_{t}}}

where \eqn{t = 1,..,T} indexes either time or cross-sectional units, \ifelse{html}{\out{Y<sub>t</sub>}}{\eqn{Y_{t}}} is the dependent variable,
\ifelse{html}{\out{P<sub>t</sub>}}{\eqn{P_{t}}} is a \code{k x 1} continuous, endogenous regressor,
\ifelse{html}{\out{&epsilon;<sub>t</sub>}}{\eqn{\epsilon_{t}}} is a structural error term with mean zero
and \ifelse{html}{\out{E(&epsilon;<sup>2</sup>)=&sigma;<sub>&epsilon;</sub><sup>2</sup>}}{\eqn{E(\epsilon^{2})=\sigma^{2}_{\epsilon}}},
\eqn{\alpha} and \ifelse{html}{\out{&beta;<sub>0</sub>}}{\eqn{\beta_0}} are model parameters.
\ifelse{html}{\out{Z;<sub>t</sub>}}{\eqn{Z_{t}}} is a \code{l x 1} vector of instruments,
and \ifelse{html}{\out{&nu;<sub>t</sub>}}{\eqn{\nu_{t}}} is a random error with mean zero and
\ifelse{html}{\out{E(&nu;<sup>2</sup>)=&sigma;<sub>&nu;</sub><sup>2</sup>}}{\eqn{E(\nu^{2}) = \sigma^{2}_{\nu}}}.
The endogeneity problem arises from the correlation of \eqn{P} and \ifelse{html}{\out{&epsilon;<sub>t</sub>}}{\eqn{\epsilon_{t}}}
through \ifelse{html}{\out{E(&epsilon;&nu;)=&sigma;<sub>&epsilon;&nu;</sub>}}{\eqn{E(\epsilon\nu) = \sigma_{\epsilon\nu}}}

\code{latentIV} considers \ifelse{html}{\out{Z<sub>t</sub>'}}{\eqn{Z_{t}^{'}}} to be a latent, discrete, exogenous variable with an unknown number of groups \eqn{m} and \eqn{\pi} is a vector of group means.
It is assumed that \eqn{Z} is independent of the error terms \eqn{\epsilon} and \eqn{\nu} and that it has at least two groups with different means.
The structural and random errors are considered normally distributed with mean zero and variance-covariance matrix \eqn{\Sigma}:
\ifelse{html}{\out{<center>&Sigma;=(&sigma;<sub>&epsilon;</sub><sup>2</sup>, &sigma;<sub>0</sub><sup>2</sup>,
<br>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&sigma;<sub>0</sub><sup>2</sup>, &sigma;<sub>&nu;</sub><sup>2</sup>)</center>}}{
\deqn{\Sigma = \left(
\begin{array}{ccc}
\sigma_{\epsilon}^{2} \& \sigma_{\epsilon\nu}\\
\sigma_{\epsilon\nu} \& \sigma_{\nu}^{2}
\end{array}\right)}}

The identification of the model lies in the assumption of the non-normality of
\ifelse{html}{\out{P<sub>t</sub>}}{\eqn{P_{t}}}, the discreteness of the unobserved instruments and the existence of
at least two groups with different means.

The method has been implemented such that the latent variable has two groups. Ebbes et al.(2005) show in a Monte Carlo experiment that
even if the true number of the categories of the instrument is larger than two, estimates are approximately consistent. Besides, overfitting in terms
of the number of groups/categories reduces the degrees of freedom and leads to efficiency loss. For a model with additional explanatory variables a Bayesian approach is needed, since
in a frequentist approach identification issues appear.

Identification of the parameters relies on the distributional assumptions of the latent instruments as well as that of
the endogenous regressor \ifelse{html}{\out{P<sub>t</sub>}}{\eqn{P_{t}}}.
Specifically, the endogenous regressor should have a non-normal distribution while the unobserved instruments, \eqn{Z}, should be discrete and have at least two groups with different means Ebbes, Wedel, and Böckenholt (2009).
A continuous distribution for the instruments leads to an unidentified model, while a normal distribution of the endogenous regressor gives rise to inefficient estimates.

Additional parameters used during model fitting and printed in \code{summary} are:
\describe{
\item{pi1}{The instrumental variables \eqn{Z} are assumed to be divided into two groups. \code{pi1} represents the estimated group mean of the first group.}
\item{pi2}{The estimated group mean of the second group of the instrumental variables \eqn{Z}.}
\item{theta5}{The probability of being in the first group of the instruments.}
\item{theta6}{The variance, \ifelse{html}{\out{&sigma;<sub>&epsilon;</sub><sup>2</sup>}}{\eqn{\sigma_{\epsilon}^{2}}}}
\item{theta7}{The covariance, \ifelse{html}{\out{&sigma;<sub>&epsilon;&nu;</sub>}}{\eqn{\sigma_{\epsilon\nu}}}}
\item{theta8}{The variance, \ifelse{html}{\out{&sigma;<sub>&nu;</sub><sup>2</sup>}}{\eqn{\sigma_{\nu}^{2}}}}
}
}
\examples{
\donttest{
data("dataLatentIV")

# function call without any initial parameter values
l  <- latentIV(y ~ P, data = dataLatentIV)
summary(l)

# function call with initial parameter values given by the user
l1 <- latentIV(y ~ P, start.params = c("(Intercept)"=2.5, P=-0.5),
               data = dataLatentIV)
summary(l1)

# use own optimization settings (see optimx())
# set maximum number of iterations to 50'000
l2 <- latentIV(y ~ P, optimx.args = list(itnmax = 50000),
               data = dataLatentIV)

# print detailed tracing information on progress
l3 <- latentIV(y ~ P, optimx.args = list(control = list(trace = 6)),
               data = dataLatentIV)

# use method L-BFGS-B instead of Nelder-Mead and print report every 50 iterations
l4 <- latentIV(y ~ P, optimx.args = list(method = "L-BFGS-B", control=list(trace = 2, REPORT=50)),
               data = dataLatentIV)

# read out all coefficients, incl auxiliary coefs
lat.all.coefs <- coef(l4)
# same as above
lat.all.coefs <- coef(l4, complete = TRUE)
# only main model coefs
lat.main.coefs <- coef(l4, complete = FALSE)
}

}
\references{
Ebbes, P., Wedel,M., Böckenholt, U., and Steerneman, A. G. M. (2005). 'Solving and Testing for Regressor-Error
(in)Dependence When no Instrumental Variables are Available: With New Evidence for the Effect of Education on Income'.
Quantitative Marketing and Economics, 3:365--392.

Ebbes P., Wedel M., Böckenholt U. (2009). “Frugal IV Alternatives to Identify the Parameter for an Endogenous Regressor.” Journal of Applied Econometrics, 24(3), 446–468.
}
\seealso{
\code{\link[REndo:summary.rendo.latent.IV]{summary}} for how fitted models are summarized

\code{\link[optimx]{optimx}} for possible elements of parameter \code{optimx.arg}
}
