#' Exporting the dataset of the queries
#'
#' This function allows you to export the previously identified queries to an xlsx file.
#'
#' @param ... List containing the data frame of queries. It may be the output of the `rd_query` or `rd_event` functions.
#' @param queries Data frame containing the identified queries. If the list is given, this argument is not required.
#' @param column Character element specifying the column containing the link of each query.
#' @param sheet_name Character element specifying the sheet name of the resulting xlsx file.
#' @param path Character element specifying the file path to save the xlsx file. If `NULL`, the file is created in the current working directory.
#' @param password String with the password to protect the worksheet and prevent others from making changes.
#' @return An xlsx file containing all the queries and, if available, hyperlinks to each one of them.
#'
#' @export

rd_export <- function(..., queries = NULL, column = NULL, sheet_name = NULL, path = NULL, password = NULL)
{
  # If the entire list resulting from the 'redcap_data' function is used
  project <- c(...)
  if(!is.null(project)){
    if(!is.null(queries)){

      warning("Queries have been specified twice so the function will not use the information in the queries argument.")

    }
    queries <- project$queries
  }

  # Making sure that the object data is a data.frame
  queries <- as.data.frame(queries)

  # Warning: links detected but the column argument is not specified.
  if (is.null(column) & !"Link" %in% names(queries) & any(queries %>% dplyr::summarise_all(~ any(grepl("https", .))))) {

    warning("A column containing links was detected. If you want to convert them into hyperlinks, please specify the column argument.", call. = F)

  }

  # Create a new workbook
  wb <- openxlsx::createWorkbook()

  # Add a new worksheet to the workbook
  sheet_name <- if (!is.null(sheet_name)) {
    sheet_name
  } else {
    "Sheet1"
  }
  sheet <- openxlsx::addWorksheet(wb, sheet_name)

  # Converting the links into hyperlinks
  if (!is.null(column) | "Link" %in% names(queries)) {

    # Stabilish the column name
    column <- if (!is.null(column)) {
      if (column %in% names(queries)) {
        column
      } else {

        # Error: the named column is not present in the dataset
        stop("The column you have specified does not exist in the data set.", call. = F)

      }
    } else {
      "Link"
    }

    # Add hyperlinks to a cell
    class(queries[, column]) <- "hyperlink"

  }

  # Write the data frame to the worksheet
  openxlsx::writeData(wb = wb, x = queries, sheet = sheet, startRow = 1, startCol = 1, rowNames = F)

  # Path to the file
  path <- if (!is.null(path)) {
    path
  } else {
    paste0(getwd(), "/example.xlsx")
  }

  # Save workbook to a file with password
  if (!is.null(password)) {

    openxlsx::protectWorksheet(wb, sheet = sheet, password = password)
    openxlsx::saveWorkbook(wb, path, overwrite = T)
    message(paste0("The file has been successfully created in '", path, "' with password protection."))

  } else {

    # Save workbook to a file
    openxlsx::saveWorkbook(wb, path, overwrite = T)
    message(paste0("The file has been successfully created in '", path, "'."))

  }
}



