% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/oc1S.R
\name{oc1S}
\alias{oc1S}
\alias{oc1S.betaMix}
\alias{oc1S.normMix}
\alias{oc1S.gammaMix}
\title{Operating Characteristics for 1 Sample Design}
\usage{
oc1S(prior, n, decision, ...)

\method{oc1S}{betaMix}(prior, n, decision, ...)

\method{oc1S}{normMix}(prior, n, decision, sigma, eps = 1e-06, ...)

\method{oc1S}{gammaMix}(prior, n, decision, eps = 1e-06, ...)
}
\arguments{
\item{prior}{Prior for analysis.}

\item{n}{Sample size for the experiment.}

\item{decision}{One-sample decision function to use; see \code{\link{decision1S}}.}

\item{...}{Optional arguments.}

\item{sigma}{The fixed reference scale. If left unspecified, the
default reference scale of the prior is assumed.}

\item{eps}{Support of random variables are determined as the
interval covering \code{1-eps} probability mass. Defaults to
\eqn{10^{-6}}.}
}
\value{
Returns a function with one argument \code{theta} which
calculates the frequency at which the decision function is
evaluated to 1 for the defined 1 sample design. Note that the
returned function takes vectors arguments.
}
\description{
The \code{oc1S} function defines a 1 sample design (prior, sample
size, decision function) for the calculation of the frequency at
which the decision is evaluated to 1 conditional on assuming
known parameters.  A function is returned which performs the actual
operating characteristics calculations.
}
\details{
The \code{oc1S} function defines a 1 sample design and
returns a function which calculates its operating
characteristics. This is the frequency with which the decision
function is evaluated to 1 under the assumption of a given true
distribution of the data defined by a known parameter
\eqn{\theta}. The 1 sample design is defined by the prior, the
sample size and the decision function, \eqn{D(y)}. These uniquely
define the decision boundary, see
\code{\link{decision1S_boundary}}.

When calling the \code{oc1S} function, then internally the critical
value \eqn{y_c} (using \code{\link{decision1S_boundary}}) is
calculated and a function is returns which can be used to
calculated the desired frequency which is evaluated as

\deqn{ F(y_c|\theta). }
}
\section{Methods (by class)}{
\itemize{
\item \code{betaMix}: Applies for binomial model with a mixture
beta prior. The calculations use exact expressions.

\item \code{normMix}: Applies for the normal model with known
standard deviation \eqn{\sigma} and a normal mixture prior for the
mean. As a consequence from the assumption of a known standard
deviation, the calculation discards sampling uncertainty of the
second moment. The function \code{oc1S} has an extra
argument \code{eps} (defaults to \eqn{10^{-6}}). The critical value
\eqn{y_c} is searched in the region of probability mass
\code{1-eps} for \eqn{y}.

\item \code{gammaMix}: Applies for the Poisson model with a gamma
mixture prior for the rate parameter.  The function
\code{oc1S} takes an extra argument \code{eps} (defaults to \eqn{10^{-6}})
which determines the region of probability mass \code{1-eps} where
the boundary is searched for \eqn{y}.
}}

\examples{

# non-inferiority example using normal approximation of log-hazard
# ratio, see ?decision1S for all details
s <- 2
flat_prior <- mixnorm(c(1,0,100), sigma=s)
nL <- 233
theta_ni <- 0.4
theta_a <- 0
alpha <- 0.05
beta  <- 0.2
za <- qnorm(1-alpha)
zb <- qnorm(1-beta)
n1 <- round( (s * (za + zb)/(theta_ni - theta_a))^2 )
theta_c <- theta_ni - za * s / sqrt(n1)

# standard NI design
decA <- decision1S(1 - alpha, theta_ni, lower.tail=TRUE)

# double criterion design
# statistical significance (like NI design)
dec1 <- decision1S(1-alpha, theta_ni, lower.tail=TRUE)
# require mean to be at least as good as theta_c
dec2 <- decision1S(0.5, theta_c, lower.tail=TRUE)
# combination
decComb <- decision1S(c(1-alpha, 0.5), c(theta_ni, theta_c), lower.tail=TRUE)

theta_eval  <- c(theta_a, theta_c, theta_ni)

# evaluate different designs at two sample sizes
designA_n1  <- oc1S(flat_prior, n1, decA)
designA_nL  <- oc1S(flat_prior, nL, decA)
designC_n1  <- oc1S(flat_prior, n1, decComb)
designC_nL  <- oc1S(flat_prior, nL, decComb)

# evaluate designs at the key log-HR of positive treatment (HR<1),
# the indecision point and the NI margin

designA_n1(theta_eval)
designA_nL(theta_eval)
designC_n1(theta_eval)
designC_nL(theta_eval)

# to understand further the dual criterion design it is useful to
# evaluate the criterions separatley:
# statistical significance criterion to warrant NI...
designC1_nL <- oc1S(flat_prior, nL, dec1)
# ... or the clinically determined indifference point
designC2_nL <- oc1S(flat_prior, nL, dec2)

designC1_nL(theta_eval)
designC2_nL(theta_eval)

# see also ?decision1S_boundary to see which of the two criterions
# will drive the decision


}
\seealso{
Other design1S: \code{\link{decision1S_boundary}},
  \code{\link{decision1S}}, \code{\link{pos1S}}
}
