% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RGWAS.normal.R
\name{RGWAS.normal}
\alias{RGWAS.normal}
\title{Perform normal GWAS (test each single SNP)}
\usage{
RGWAS.normal(
  pheno,
  geno,
  ZETA = NULL,
  package.MM = "gaston",
  covariate = NULL,
  covariate.factor = NULL,
  structure.matrix = NULL,
  n.PC = 0,
  min.MAF = 0.02,
  P3D = TRUE,
  n.core = 1,
  parallel.method = "mclapply",
  sig.level = 0.05,
  method.thres = "BH",
  plot.qq = TRUE,
  plot.Manhattan = TRUE,
  plot.method = 1,
  plot.col1 = c("dark blue", "cornflowerblue"),
  plot.col2 = 1,
  plot.type = "p",
  plot.pch = 16,
  saveName = NULL,
  main.qq = NULL,
  main.man = NULL,
  plot.add.last = FALSE,
  return.EMM.res = FALSE,
  optimizer = "nlminb",
  thres = TRUE,
  skip.check = FALSE,
  verbose = TRUE,
  verbose2 = FALSE,
  count = TRUE,
  time = TRUE
)
}
\arguments{
\item{pheno}{Data frame where the first column is the line name (gid). The remaining columns should be a phenotype to test.}

\item{geno}{Data frame with the marker names in the first column. The second and third columns contain the chromosome and map position.
Columns 4 and higher contain the marker scores for each line, coded as {-1, 0, 1} = {aa, Aa, AA}.}

\item{ZETA}{A list of covariance (relationship) matrix (K: \eqn{m \times m}) and its design matrix (Z: \eqn{n \times m}) of random effects.
Please set names of list "Z" and "K"! You can use more than one kernel matrix.
For example,

ZETA = list(A = list(Z = Z.A, K = K.A), D = list(Z = Z.D, K = K.D))
\describe{
\item{Z.A, Z.D}{Design matrix (\eqn{n \times m}) for the random effects. So, in many cases, you can use the identity matrix.}
\item{K.A, K.D}{Different kernels which express some relationships between lines.}
}
For example, K.A is additive relationship matrix for the covariance between lines, and K.D is dominance relationship matrix.}

\item{package.MM}{The package name to be used when solving mixed-effects model. We only offer the following three packages: 
"RAINBOWR", "MM4LMM" and "gaston". Default package is `gaston`.
See more details at \code{\link{EM3.general}}.}

\item{covariate}{A \eqn{n \times 1} vector or a \eqn{n \times p _ 1} matrix. You can insert continuous values, such as other traits or genotype score for special markers.
This argument is regarded as one of the fixed effects.}

\item{covariate.factor}{A \eqn{n \times p _ 2} dataframe. You should assign a factor vector for each column.
Then RGWAS changes this argument into model matrix, and this model matrix will be included in the model as fixed effects.}

\item{structure.matrix}{You can use structure matrix calculated by structure analysis when there are population structure.
You should not use this argument with n.PC > 0.}

\item{n.PC}{Number of principal components to include as fixed effects. Default is 0 (equals K model).}

\item{min.MAF}{Specifies the minimum minor allele frequency (MAF).
If a marker has a MAF less than min.MAF, it is assigned a zero score.}

\item{P3D}{When P3D = TRUE, variance components are estimated by REML only once, without any markers in the model.
When P3D = FALSE, variance components are estimated by REML for each marker separately.}

\item{n.core}{Setting n.core > 1 will enable parallel execution on a machine with multiple cores. 
This argument is not valid when `parallel.method = "furrr"`.}

\item{parallel.method}{Method for parallel computation. We offer three methods, "mclapply", "furrr", and "foreach". 

When `parallel.method = "mclapply"`, we utilize \code{\link[pbmcapply]{pbmclapply}} function in the `pbmcapply` package 
with `count = TRUE` and \code{\link[parallel]{mclapply}} function in the `parallel` package with `count = FALSE`. 

When `parallel.method = "furrr"`, we utilize \code{\link[furrr]{future_map}} function in the `furrr` package. 
With `count = TRUE`, we also utilize \code{\link[progressr]{progressor}} function in the `progressr` package to show the progress bar, 
so please install the `progressr` package from github (\url{https://github.com/HenrikBengtsson/progressr}). 
For `parallel.method = "furrr"`, you can perform multi-thread parallelization by 
sharing memories, which results in saving your memory, but quite slower compared to `parallel.method = "mclapply"`.

When `parallel.method = "foreach"`, we utilize \code{\link[foreach]{foreach}} function in the `foreach` package 
with the utilization of \code{\link[parallel]{makeCluster}} function in `parallel` package, 
and \code{\link[doParallel]{registerDoParallel}} function in `doParallel` package. 
With `count = TRUE`, we also utilize \code{\link[utils]{setTxtProgressBar}} and 
\code{\link[utils]{txtProgressBar}} functions in the `utils` package to show the progress bar.

We recommend that you use the option `parallel.method = "mclapply"`, but for Windows users, 
this parallelization method is not supported. So, if you are Windows user, 
we recommend that you use the option `parallel.method = "foreach"`.}

\item{sig.level}{Significance level for the threshold. The default is 0.05.}

\item{method.thres}{Method for detemining threshold of significance. "BH" and "Bonferroni are offered.}

\item{plot.qq}{If TRUE, draw qq plot.}

\item{plot.Manhattan}{If TRUE, draw manhattan plot.}

\item{plot.method}{If this argument = 1, the default manhattan plot will be drawn.
If this argument = 2, the manhattan plot with axis based on Position (bp) will be drawn.
 Also, this plot's color is changed by all chromosomes.}

\item{plot.col1}{This argument determines the color of the manhattan plot.
You should substitute this argument as color vector whose length is 2.
plot.col1[1] for odd chromosomes and plot.col1[2] for even chromosomes}

\item{plot.col2}{Color of the manhattan plot. color changes with chromosome and it starts from plot.col2 + 1
(so plot.col2 = 1 means color starts from red.)}

\item{plot.type}{This argument determines the type of the manhattan plot. See the help page of "plot".}

\item{plot.pch}{This argument determines the shape of the dot of the manhattan plot. See the help page of "plot".}

\item{saveName}{When drawing any plot, you can save plots in png format. In saveName, you should substitute the name you want to save.
When saveName = NULL, the plot is not saved.}

\item{main.qq}{The title of qq plot. If this argument is NULL, trait name is set as the title.}

\item{main.man}{The title of manhattan plot. If this argument is NULL, trait name is set as the title.}

\item{plot.add.last}{If saveName is not NULL and this argument is TRUE, then you can add lines or dots to manhattan plots.
However, you should also write "dev.off()" after adding something.}

\item{return.EMM.res}{When return.EMM.res = TRUE, the results of equation of mixed models are included in the result of RGWAS.}

\item{optimizer}{The function used in the optimization process. 
We offer "optim", "optimx", and "nlminb" functions. 
This argument is only valid when `package.MM = 'RAINBOWR'`.}

\item{thres}{If thres = TRUE, the threshold of the manhattan plot is included in the result of RGWAS.
When return.EMM.res or thres is TRUE, the results will be "list" class.}

\item{skip.check}{As default, RAINBOWR checks the type of input data and modifies it into the correct format. 
However, it will take some time, so if you prepare the correct format of input data, you can skip this procedure 
by setting `skip.check = TRUE`.}

\item{verbose}{If this argument is TRUE, messages for the current steps will be shown.}

\item{verbose2}{If this argument is TRUE, welcome message will be shown.}

\item{count}{When count is TRUE, you can know how far RGWAS has ended with percent display.}

\item{time}{When time is TRUE, you can know how much time it took to perform RGWAS.}
}
\value{
\describe{
\item{$D}{Dataframe which contains the information of the map you input and the results of RGWAS (-log10(p)) which correspond to the map.}
\item{$thres}{A vector which contains the information of threshold determined by FDR = 0.05.}
\item{$EMM.res}{This output is a list which contains the information about the results of "EMM" perfomed at first in regular GWAS.
 If you want to know details, see the description for the function "EMM1" or "EMM2".}
}
}
\description{
This function performs single-SNP GWAS (genome-wide association studies). The model of GWAS is

\deqn{y = X \beta + S _ {i} \alpha _ {i} + Q v +  Z u + \epsilon,}

where \eqn{y} is the vector of phenotypic values,
\eqn{X \beta}, \eqn{S _ {i} \alpha _ {i}}, \eqn{Q v} are the terms of fixed effects,
\eqn{Z u} is the term of random effects and \eqn{e} is the vector of residuals.
\eqn{X \beta} indicates all of the fixed effects other than the effect of SNPs
to be tested and of population structure, and often this term also plays
a role as an intercept. For \eqn{S _ {i} \alpha _ {i}}, \eqn{S _ {i}}
is the ith marker of genotype data and \eqn{\alpha _ {i}} is the effect of that marker.
\eqn{Q v} is the term to correct the effect of population structure.
\eqn{Z u} is the term of polygenetic effects, and suppose that \eqn{u}
follows the multivariate normal distribution whose variance-covariance
 matrix is the genetic covariance matrix. \eqn{u \sim MVN (0, K \sigma_{u}^{2})}.
Finally, the residual term is assumed to identically and independently follow
 a normal distribution as shown in the following equation.
 \eqn{e \sim MVN (0, I \sigma_{e}^{2})}.
}
\details{
P-value for each marker is calculated by performing F-test
against the F-value as follows (Kennedy et al., 1992).

\deqn{F = \frac { ( L' \hat { b } )' [ L' ( X' H ^ { - 1 } X ) ^ { - 1 }
L ] ^ { - 1 } ( L' \hat { b } ) } { f \hat { \sigma }_ { u } ^ { 2 } },}

where \eqn{b} is the vector of coefficients of the fixed effects, which combines
\eqn{\beta}, \eqn{\alpha _ {i}}, \eqn{v} in the horizontal direction and \eqn{L}
is a matrix to indicate which effects in \eqn{b} are tested.
\eqn{H} is calculated by dividing the estimated variance-covariance
matrix for the phenotypic values by \eqn{\sigma _ { u } ^ { 2 }},
and is calculated by \eqn{H = Z K Z' + \hat{\lambda} I}.
\eqn{\hat{\lambda}} is the maximum likelihood estimator
of the ratio between the residual variance and the additive genetic variance.
\eqn{\hat{b}} is the maximum likelihood estimator of \eqn{b}
and is calculated by \eqn{\hat { b } = ( X' H ^ { - 1 } X ) ^ { - 1 } X' H ^ { - 1 } y }.
\eqn{f} is the number of the fixed effects to be tested, and
\eqn{\hat { \sigma }_ { u } ^ { 2 }} is estimated by the following formula.
\deqn{\hat { \sigma }_ { u } ^ { 2 } = \frac { ( y - X  \hat { b } )' H ^ { - 1 } ( y - X  \hat { b } ) } { n - p },}
where \eqn{n} is the sample size and \eqn{p} is the number of the all fixed effects.
We calculated each p-value using the fact that the above F-value follows
the F distribution with the degree of freedom (\eqn{f},\eqn{n - p}).
}
\examples{
\dontshow{
  ### Import RAINBOWR
  require(RAINBOWR)

  ### Load example datasets
  data("Rice_Zhao_etal")
  Rice_geno_score <- Rice_Zhao_etal$genoScore
  Rice_geno_map <- Rice_Zhao_etal$genoMap
  Rice_pheno <- Rice_Zhao_etal$pheno


  ### Select one trait for example
  trait.name <- "Flowering.time.at.Arkansas"
  y <- as.matrix(Rice_pheno[1:30, trait.name, drop = FALSE])
  # use first 30 accessions

  ### Remove SNPs whose MAF <= 0.05
  x.0 <- t(Rice_geno_score)
  MAF.cut.res <- MAF.cut(x.0 = x.0, map.0 = Rice_geno_map)
  x <- MAF.cut.res$x
  map <- MAF.cut.res$map


  ### Estimate genomic relationship matrix (GRM)
  K.A <- calcGRM(genoMat = x)


  ### Modify data
  modify.data.res <- modify.data(pheno.mat = y, geno.mat = x, map = map,
                                 return.ZETA = TRUE, return.GWAS.format = TRUE)
  pheno.GWAS <- modify.data.res$pheno.GWAS
  geno.GWAS <- modify.data.res$geno.GWAS
  ZETA <- modify.data.res$ZETA



  ### Perform single-SNP GWAS
  normal.res <- RGWAS.normal(pheno = pheno.GWAS, geno = geno.GWAS,
                             ZETA = ZETA, n.PC = 4, P3D = TRUE,
                             plot.qq = FALSE, plot.Manhattan = FALSE, verbose = FALSE,
                             verbose2 = FALSE, count = FALSE, time = FALSE,
                             package.MM = "gaston", parallel.method = "mclapply",
                             skip.check = TRUE, n.core = 1)
}


\donttest{
  ### Import RAINBOWR
  require(RAINBOWR)

  ### Load example datasets
  data("Rice_Zhao_etal")
  Rice_geno_score <- Rice_Zhao_etal$genoScore
  Rice_geno_map <- Rice_Zhao_etal$genoMap
  Rice_pheno <- Rice_Zhao_etal$pheno

  ### View each dataset
  See(Rice_geno_score)
  See(Rice_geno_map)
  See(Rice_pheno)

  ### Select one trait for example
  trait.name <- "Flowering.time.at.Arkansas"
  y <- as.matrix(Rice_pheno[, trait.name, drop = FALSE])

  ### Remove SNPs whose MAF <= 0.05
  x.0 <- t(Rice_geno_score)
  MAF.cut.res <- MAF.cut(x.0 = x.0, map.0 = Rice_geno_map)
  x <- MAF.cut.res$x
  map <- MAF.cut.res$map


  ### Estimate genomic relationship matrix (GRM)
  K.A <- calcGRM(genoMat = x)


  ### Modify data
  modify.data.res <- modify.data(pheno.mat = y, geno.mat = x, map = map,
                                 return.ZETA = TRUE, return.GWAS.format = TRUE)
  pheno.GWAS <- modify.data.res$pheno.GWAS
  geno.GWAS <- modify.data.res$geno.GWAS
  ZETA <- modify.data.res$ZETA


  ### View each data for RAINBOWR
  See(pheno.GWAS)
  See(geno.GWAS)
  str(ZETA)



  ### Perform single-SNP GWAS
  normal.res <- RGWAS.normal(pheno = pheno.GWAS, geno = geno.GWAS,
                             ZETA = ZETA, n.PC = 4, P3D = TRUE,
                             package.MM = "gaston", parallel.method = "mclapply",
                             skip.check = TRUE, n.core = 2)
  See(normal.res$D)  ### Column 4 contains -log10(p) values for markers
}
}
\references{
Kennedy, B.W., Quinton, M. and van Arendonk, J.A. (1992) Estimation of effects of single genes on quantitative traits. J Anim Sci. 70(7): 2000-2012.

Storey, J.D. and Tibshirani, R. (2003) Statistical significance for genomewide studies. Proc Natl Acad Sci. 100(16): 9440-9445.

Yu, J. et al. (2006) A unified mixed-model method for association mapping that accounts for multiple levels of relatedness. Nat Genet. 38(2): 203-208.

Kang, H.M. et al. (2008) Efficient Control of Population Structure in Model Organism Association Mapping. Genetics. 178(3): 1709-1723.

Kang, H.M. et al. (2010) Variance component model to account for sample structure in genome-wide association studies. Nat Genet. 42(4): 348-354.

Zhang, Z. et al. (2010) Mixed linear model approach adapted for genome-wide association studies. Nat Genet. 42(4): 355-360.

Endelman, J.B. (2011) Ridge Regression and Other Kernels for Genomic Selection with R Package rrBLUP. Plant Genome J. 4(3): 250.

Endelman, J.B. and Jannink, J.L. (2012) Shrinkage Estimation of the Realized Relationship Matrix. G3 Genes, Genomes, Genet. 2(11): 1405-1413.

Zhou, X. and Stephens, M. (2012) Genome-wide efficient mixed-model analysis for association studies. Nat Genet. 44(7): 821-824.
}
