% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/processPgDetections.R
\name{processPgDetections}
\alias{processPgDetections}
\title{Load and Process Detections from Pamguard}
\usage{
processPgDetections(
  pps,
  mode = c("db", "time", "recording"),
  id = NULL,
  grouping = NULL,
  format = c("\%m/\%d/\%Y \%H:\%M:\%OS", "\%m-\%d-\%Y \%H:\%M:\%OS",
    "\%Y/\%m/\%d \%H:\%M:\%OS", "\%Y-\%m-\%d \%H:\%M:\%OS"),
  progress = TRUE,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{pps}{a \linkS4class{PAMpalSettings} object containing the databases,
binaries, and functions to use for processing data. See
\code{\link[PAMpal]{PAMpalSettings}}. Can also be an \linkS4class{AcousticStudy}
object, in which case the \code{pps} slot will be used.}

\item{mode}{selector for how to organize your data in to events. \code{db}
will organize by events based on tables in the databases. \code{time}
will organize into events based on timestamps provided in \code{grouping}.
\code{recording} will organize events by the start and end times of recording
files found in the database. For \code{time} and \code{recording}, ALL detections
between the start and end times are included, for \code{db} only selected
detections are included.}

\item{id}{an event name or id for this study, will default to today's date if
not supplied (recommended to supply your own informative id)}

\item{grouping}{For \code{mode = 'db'}, the table to group events by.
  Either \code{event} to use the OfflineEvents table, or \code{detGroup} to
  use the detection group localiser module groups.

  For \code{mode = 'time'},
  this should be a data frame with three mandatory columns and 1 row
  for each separate event. The mandatory columns are \code{start}, \code{end},
  and \code{id}. \code{start} and \code{end} should specify the
  start and end time of the event and must be in UTC. \code{id} should
  specify a unique id for each event. There are also optional columns
  \code{species}, \code{db}, and \code{sr}. \code{species} should provide a
  species ID if it is available. \code{db} and \code{sr} are the corresponding
  database and sample rate to associate with a particular event, these typically
  do not need to be specified as the function will attempt to automatically match
  them based on the times of the events and the databases. Note that \code{db}
  must be the full filepath to the database. If a clear match is not found then
  the user will be prompted to either select from a list or input the proper
  sample rate.

  \code{grouping} can be supplied either as a data frame or as
  a filepath to a csv file.}

\item{format}{the date format for the \code{start} and \code{end} columns
in \code{grouping} if it is a csv. Times are assumed to be UTC. See
details section of \link{strptime} for more information on how to properly
format this}

\item{progress}{logical flog to show progress bars}

\item{verbose}{logical flag to show messages}

\item{\dots}{additional arguments to pass onto to different methods}
}
\value{
an \linkS4class{AcousticStudy} object with one \linkS4class{AcousticEvent}
  for each event in the \code{events} slot, and the \linkS4class{PAMpalSettings} object
  used stored in the \code{pps} slot.
}
\description{
Loads and processes acoustic detection data that has been
  run through Pamguard. Uses the binary files and database(s) contained
  in \code{pps}, and will group your data into events by the
  grouping present in the 'OfflineEvents' and 'Detection Group Localiser'
  tables (\code{mode = 'db'}) or by the grouping specified by start and end
  times in the supplied \code{grouping} (\code{mode = 'time'}), or by start and
  end of recording files (\code{mode = 'recording'}). Will apply
  all processing functions in \code{pps} to the appropriate modules
}
\details{
If \code{mode} is not specified, it will try to be automatically determined
  in the following order. 1) if a \code{grouping} data.frame or CSV is provided, then
  \code{mode='time'} will be used. 2) If there are labelled events present in the
  database, \code{mode='db'} will be used. 3) \code{mode='recording'} will be used,
  which should be equivalent to loading all possible data.
}
\examples{

exPps <- new('PAMpalSettings')
exPps <- addDatabase(exPps, system.file('extdata', 'Example.sqlite3', package='PAMpal'))
exPps <- addBinaries(exPps, system.file('extdata', 'Binaries', package='PAMpal'))
exClick <- function(data) {
    standardClickCalcs(data, calibration=NULL, filterfrom_khz = 0)
}
exPps <- addFunction(exPps, exClick, module = 'ClickDetector')
exPps <- addFunction(exPps, roccaWhistleCalcs, module='WhistlesMoans')
exPps <- addFunction(exPps, standardCepstrumCalcs, module = 'Cepstrum')
# process events labelled within the Pamguard database
exStudyDb <- processPgDetections(exPps, mode='db', id='Example')
# can also give an AcousticStudy as input and it will use same functions and data
reprocess <- processPgDetections(exStudyDb, mode='db', id='Reprocess')
# process events with manually set start/end times
grp <- data.frame(start = as.POSIXct('2018-03-20 15:25:10', tz='UTC'),
                  end = as.POSIXct('2018-03-20 15:25:11', tz='UTC'),
                  id = 'GroupExample')
exStudyTime <- processPgDetections(exPps, mode='time', grouping=grp, id='Time')
# process events by recording event
exStudyRecording <- processPgDetections(exPps, mode='recording', id='Recording')

}
\author{
Taiki Sakai \email{taiki.sakai@noaa.gov}
}
