#' @title feature_distribution
#'
#' @description Function collects data from STRING database, scales, as well as prepares additional score integration to visualise data feature distribution.
#'
#' @param  data Requires a  data frame generated by score_genes function; class - data frame
#' @return multiple summary plots; class - plots
#'
#' @importFrom ggExtra ggMarginal
#' @importFrom  dendextend find_k
#' @importFrom  dendextend color_labels
#' @importFrom  dendextend  color_branches
#' @importFrom  dendextend   colored_bars
#' @import cluster
#' @importFrom ggplot2 aes
#' @importFrom ggplot2 theme
#' @ImpportFrom ggplot2 element_text
#' @ImportFrom ggplot2 geom_col
#' @ImportFrom ggplot2 ggplot
#' @import RColorBrewer
#' @importFrom graphics legend
#' @import utils
#' @importFrom stats  hclust
#' @importFrom stats as.dendrogram
#' @importFrom  RCurl getURL
#' @export
#' @examples
#'  \dontrun{
#' path_to_test_data<- system.file("extdata", "test_data.tabular", package="OmicInt")
#' # basic usage of feature_distribution
#' df<-utils::read.table(path_to_test_data)
#' feature_distribution(df)}
#' @export
feature_distribution<-function(data){

  #download the data from curated databases

  #add location associations
  location_url <- RCurl::getURL("https://gitlab.com/Algorithm379/databases/-/raw/main/Subcellular.locationmerged_protein_data.csv")
  location_df <- utils::read.csv(text = location_url)

   data$"Location"<-ifelse(data$"Symbol"%in%location_df$"Symbol",location_df$"Subcellular.location","NA")
  #rename NAs
  data$"Location"<-ifelse( is.na(data$"Location"),"NA", data$"Location")

  #add class associations
  classes_url <- RCurl::getURL("https://gitlab.com/Algorithm379/databases/-/raw/main/HS_protein_classes_curated.csv")
  classes <- utils::read.csv(text = classes_url)

  #prepare data frame
  data$"Class"<-ifelse(data$"Symbol"%in%classes$"Gene",classes$"Class","NA")

  #prepare plot

  df<-data[,c("Interactors","log2FoldChange","Association_score","Specificity_score","LFCscore", "Class","Location")]

  rownames(df)<-data$"Symbol"

  #prepare colors
  qual_col_pals <- RColorBrewer::brewer.pal.info[which(RColorBrewer::brewer.pal.info$"category"%in%c('qual')),] #max number of colours 335, setting for qual gives 74
  seq_col_pals <- RColorBrewer::brewer.pal.info[which(RColorBrewer::brewer.pal.info$"category"%in%c('seq')),]

  col_vector <- unlist(mapply(RColorBrewer::brewer.pal, qual_col_pals$"maxcolors", rownames(qual_col_pals)))

  seq_col_vector <- as.vector(mapply(RColorBrewer::brewer.pal, seq_col_pals$"maxcolors", rownames(seq_col_pals)))

  col_colors <- col_vector[1:nlevels(as.factor(data$"Location"))]
  names(col_colors)<-levels(as.factor(data$"Location"))


  seq_colors <- seq_col_vector[1:nlevels(as.factor(data$"Class"))]
  names(seq_colors)<-levels(as.factor(data$"Class"))

  #set condition for df
 loc_vector_names<-c()
 class_vector_names<-c()
  for(name in rownames(df)){

    loc_col<-data[which(name==data$"Symbol"),"Location"]
    class_col<-data[which(name==data$"Symbol"),"Class"]

    class_vector_names<-c(class_vector_names,seq_colors[class_col])
    loc_vector_names<-c(loc_vector_names,col_colors[loc_col])
  }

 col_df<-cbind(class_vector_names,loc_vector_names)
 rownames(col_df)<-data$"Symbol"
 colnames(col_df)<-c("Class_col","Location_col")
 df<-cbind(df,col_df)


 # to avoid conflicts and warning messages
 Class<-df$"Class"
 Location<-df$"Location"
 Interactors<-df$"Interactors"
 Association_score<-df$"Association_score"
 Specificity_score<-df$"Specificity_score"
 LFCscore<-df$"LFCscore"


 p <- ggplot2::ggplot(df) +ggplot2::geom_point(ggplot2::aes(x = LFCscore, y = Interactors, color=Association_score))
 ggExtra::ggMarginal(p, type = "densigram",
            size = 3, fill="lightblue")

 p <- ggplot2::ggplot(df, ggplot2::aes(x = LFCscore, y = Interactors, color=Specificity_score)) +
   ggplot2::geom_point()
 ggExtra::ggMarginal(p, type = "densigram",
                     size = 3, fill="lightblue")

 p <- ggplot2::ggplot(df, ggplot2::aes(x = LFCscore, y = Interactors, color=Location)) +
   ggplot2::geom_point()
 ggExtra::ggMarginal(p, type = "densigram",
                     size = 3, fill="lightblue")

 p <- ggplot2::ggplot(df, ggplot2::aes(x = LFCscore, y = Interactors, color=Class)) +
   ggplot2::geom_point()
ggExtra::ggMarginal(p, type = "density",
                     size = 3,groupColour = TRUE,
                     groupFill = TRUE)


}



