#' @title Import data files from Oregon RFID antenna readers
#' @description Imports files from Oregon RFID (radio-frequency identification) ORMR (Oregon RFID Multi-Reader) and ORSR (Oregon RFID Single Reader) antenna readers. Only detections are retained during compilation (events are removed).
#' 
#' This function will only work with raw data downloaded directly from Oregon RFID stationary readers. The files must be delimited (tab, comma, or semicolon) and unedited by the user.
#' @param file .txt file generated by an Oregon RFID reader.
#' @param delim field/column delimiter, which must be tab ('\\t'), comma (',') or semi-colon (';').
#' @param verbose If \emph{TRUE}, a description of the compiled data is printed to the console.
#' @details The field/column delimiter must be tab, comma or semi-colon for data compilation and further analysis. The function cannot be used for space delimited data. 
#' 
#' The tag number column (\emph{TAG}) is required for subsequent analyses, and the function will return a warning if \emph{TAG} is not included in the data file. 
#' 
#' Note that corruption may occur in reader data files. Check your data files and compiled data carefully to ensure accuracy.
#' @return Returns a tibble object of data compiled from a single ORFID data file.
#' @author Hugo Marques <biohmarques@@gmail.com>, Annika Putt <annika@@instream.net>
#' @importFrom magrittr %>%
#' @export
#' @examples
#' 
#' # Import a single tab-deliminated data file from an ORFID reader
#' import_ORFID(file = system.file("extdata/orfid_data_file.txt", package = "ORFID"), delim = "\t")
#' 

import_ORFID <- function(file, delim, verbose = FALSE) {
    
    if (!(delim %in% c("\t", ",", ";"))) {
        stop("Column delimiter must be tab, comma, or semicolon")
    }
    
    raw_data <- suppressWarnings(
        readr::read_delim(file, 
                          delim = delim, 
                          skip = grep("* records ---------$", readLines(file, warn = FALSE)),
                          show_col_types = FALSE)
    )
    
    if (ncol(raw_data) == 1) {
        stop("Column delimiter must be tab, comma, or semicolon")
    }
    
    raw_data <- raw_data %>%
        dplyr::filter(DTY == "S" | DTY == "I")
    
    if (("DUR" %in% names(raw_data))) {
        raw_data <- raw_data %>%
            dplyr::mutate(DUR = as.character(DUR)) %>%
            dplyr::mutate(DUR = readr::parse_time(DUR, '%H:%M:%OS'))
    }
    
    if (("SCD" %in% names(raw_data))) {
        raw_data <- raw_data %>%
            dplyr::mutate(SCD = as.factor(SCD))
    }
    
    
    if (!("SCD" %in% names(raw_data))) {
        warning("Site code (SCD) is required for further analysis.")
    }
    
    if (("TTY" %in% names(raw_data))) {
        raw_data <- raw_data %>%
            dplyr::mutate(TTY = as.factor(TTY))
    }
    
    if (!("TAG" %in% names(raw_data))) {
        warning("Tag number (TAG) is required for further analysis.")
    }
    
    raw_data <- raw_data %>% 
        dplyr::distinct()
    
    if (verbose == TRUE) {
        return(dplyr::glimpse(raw_data))
    } else {
        return(raw_data)
    }
    
}
