#' Local search Operators
#'
#' Local search operators for the MOEA/D
#'
#' This routine calls the local search operator for the MOEADr package, as part
#' of the call to [perform_variation()]. This operator requires its entry
#' in the variation stack (see Section `Variation Operators` of [moead()])
#' to contain the following fields:
#' - `name = "localsearch"`
#' - `type` (see [get_localsearch_methods()] for details)
#' - `gamma.ls` (optional): probability of application of local search to a
#' given subproblem at any given iteration (numeric between 0 and 1)
#' - `tau.ls` (optional): period of application of local search to each
#' subproblem (positive integer)
#' - `trunc.x` (optional): logical flag for truncating the results of the
#' local search operator to the limits defined by `problem$xmin`,
#' `problem$xmax` (logical). Defaults to `TRUE`.
#'
#' Whenever local search is triggered for a given subproblem, it cancels
#' all other variation operators _for that subproblem_ and is executed directly
#' on the incumbent solution.
#'
#' This routine is intended to be used internally by [perform_variation()],
#' and should not be called directly by the user.
#'
#' @param ... arguments to be passed down to the specific `ls_`**xyz**`()`
#' functions. A list of available local search methods can be generated by
#' [get_localsearch_methods()]. Consult the documentation of the specific
#' functions for details.
#'
#' @return Either a matrix `Xls` containing the modified points (points
#' that did not undergo local search are indicated as NA in this output matrix),
#' or a list object containing the `Xls` matrix and an integer `nfe`, informing
#' how many additional function evaluations were performed by the local search
#' operator. The specific output is defined by the `ls_`**xyz**`()` method used.
#'
#' @export
#'
#' @section References:
#' F. Campelo, L.S. Batista, C. Aranha (2020): The {MOEADr} Package: A
#' Component-Based Framework for Multiobjective Evolutionary Algorithms Based on
#' Decomposition. Journal of Statistical Software \doi{10.18637/jss.v092.i06}\cr

variation_localsearch <- function(...){

  # ========== Error catching and default value definitions
  # All error catching and default value definitions are assumed to have been
  # verified in the calling function perform_variation().
  # ==========

  # prepare arguments for calling ls_xyz() function
  # vls.input.pars <- c(as.list(environment()), ls.args) # <------ for debugging
  vls.input.pars <- as.list(sys.call())[-1]

  # Call local search method specified in "type"
  opname <- paste0("ls_", vls.input.pars$ls.args$type)
  Xls    <- do.call(opname,
                    args = vls.input.pars)

  # Truncate if required
  if(vls.input.pars$ls.args$trunc.x) {
    if(is.matrix(Xls)) {
      Xls <- matrix(pmax(0, pmin(Xls, 1)),
                    nrow  = nrow(Xls),
                    byrow = FALSE)
    } else {
      Xls$X <- matrix(pmax(0, pmin(Xls$X, 1)),
                      nrow  = nrow(Xls$X),
                      byrow = FALSE)
    }
  }

  # Return
  return(Xls)

}
