% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/analyse_pIRIRSequence.R
\name{analyse_pIRIRSequence}
\alias{analyse_pIRIRSequence}
\title{Analyse post-IR IRSL measurement sequences}
\usage{
analyse_pIRIRSequence(object, signal.integral.min, signal.integral.max,
  background.integral.min, background.integral.max, dose.points = NULL,
  sequence.structure = c("TL", "IR50", "pIRIR225"), plot = TRUE,
  plot.single = FALSE, ...)
}
\arguments{
\item{object}{\linkS4class{RLum.Analysis} or \link{list} of \linkS4class{RLum.Analysis} objects (\strong{required}):
input object containing data for analysis.
If a \link{list} is provided the functions tries to iteratre over the list.}

\item{signal.integral.min}{\link{integer} (\strong{required}):
lower bound of the signal integral. Provide this value as vector for different
integration limits for the different IRSL curves.}

\item{signal.integral.max}{\link{integer} (\strong{required}):
upper bound of the signal integral. Provide this value as vector for different
integration limits for the different IRSL curves.}

\item{background.integral.min}{\link{integer} (\strong{required}):
lower bound of the background integral. Provide this value as vector for
different integration limits for the different IRSL curves.}

\item{background.integral.max}{\link{integer} (\strong{required}):
upper bound of the background integral. Provide this value as vector for
different integration limits for the different IRSL curves.}

\item{dose.points}{\link{numeric} (\emph{optional}):
a numeric vector containing the dose points values. Using this argument overwrites dose point
values in the signal curves.}

\item{sequence.structure}{\link{vector} \link{character} (\emph{with default}):
specifies the general sequence structure. Allowed values are \code{"TL"} and
any \code{"IR"} combination (e.g., \code{"IR50"},\code{"pIRIR225"}).
Additionally a parameter \code{"EXCLUDE"} is allowed to exclude curves from
the analysis (Note: If a preheat without PMT measurement is used, i.e.
preheat as none TL, remove the TL step.)}

\item{plot}{\link{logical} (\emph{with default}):
enables or disables plot output.}

\item{plot.single}{\link{logical} (\emph{with default}):
single plot output (\code{TRUE/FALSE}) to allow for plotting the results in single plot
windows. Requires \code{plot = TRUE}.}

\item{...}{further arguments that will be passed to the function
\link{analyse_SAR.CWOSL} and \link{plot_GrowthCurve}. Furthermore, the arguments \code{main} (headers), \code{log} (IRSL curves), \code{cex} (control
the size) and \code{mtext.outer} (additional text on the plot area) can be passed to influence the plotting. If the input
is list, \code{main} can be passed as \link{vector} or \link{list}.}
}
\value{
Plots (\emph{optional}) and an \linkS4class{RLum.Results} object is
returned containing the following elements:

\tabular{lll}{
\strong{DATA.OBJECT} \tab \strong{TYPE} \tab \strong{DESCRIPTION} \cr
\code{..$data} : \tab  \code{data.frame} \tab Table with De values \cr
\code{..$LnLxTnTx.table} : \tab \code{data.frame} \tab with the LnLxTnTx values \cr
\code{..$rejection.criteria} : \tab \link{data.frame} \tab rejection criteria \cr
\code{..$Formula} : \tab \link{list} \tab Function used for fitting of the dose response curve \cr
\code{..$call} : \tab \link{call} \tab the original function call
}

The output should be accessed using the function \link{get_RLum}.
}
\description{
The function performs an analysis of post-IR IRSL sequences including curve
fitting on \linkS4class{RLum.Analysis} objects.
}
\details{
To allow post-IR IRSL protocol (Thomsen et al., 2008) measurement analyses
this function has been written as extended wrapper function for the function
\link{analyse_SAR.CWOSL}, facilitating an entire sequence analysis in
one run. With this, its functionality is strictly limited by the
functionality of the function \link{analyse_SAR.CWOSL}.

**Defining the sequence structure **

The argument \code{sequence.structure} expects a shortened pattern of your sequence structure and was
mainly introduced to ease the use of the function. For example: If your measurement data contains
the following curves: \code{TL}, \code{IRSL}, \code{IRSL}, \code{TL}, \code{IRSL}, \code{IRSL}, the sequence pattern in \code{sequence.structure}
becomes \code{c('TL', 'IRSL', 'IRSL')}. The second part of your sequence for one cycle should be
similar and can be discarded. If this is not the case (e.g., additional hotbleach) such curves
have to be removed before using the function.

\strong{If the input is a \code{list}}

If the input is a list of RLum.Analysis-objects, every argument can be provided as list to allow
for different sets of parameters for every single input element.
For further information see \link{analyse_SAR.CWOSL}.
}
\note{
Best graphical output can be achieved by using the function \code{pdf}
with the following options:

\code{pdf(file = "<YOUR FILENAME>", height = 15, width = 15)}
}
\section{Function version}{
 0.2.4
}

\examples{


### NOTE: For this example existing example data are used. These data are non pIRIR data.
###
##(1) Compile example data set based on existing example data (SAR quartz measurement)
##(a) Load example data
data(ExampleData.BINfileData, envir = environment())

##(b) Transform the values from the first position in a RLum.Analysis object
object <- Risoe.BINfileData2RLum.Analysis(CWOSL.SAR.Data, pos=1)

##(c) Grep curves and exclude the last two (one TL and one IRSL)
object <- get_RLum(object, record.id = c(-29,-30))

##(d) Define new sequence structure and set new RLum.Analysis object
sequence.structure  <- c(1,2,2,3,4,4)
sequence.structure <- as.vector(sapply(seq(0,length(object)-1,by = 4),
                                       function(x){sequence.structure + x}))

object <-  sapply(1:length(sequence.structure), function(x){

  object[[sequence.structure[x]]]

})

object <- set_RLum(class = "RLum.Analysis", records = object, protocol = "pIRIR")

##(2) Perform pIRIR analysis (for this example with quartz OSL data!)
## Note: output as single plots to avoid problems with this example
results <- analyse_pIRIRSequence(object,
     signal.integral.min = 1,
     signal.integral.max = 2,
     background.integral.min = 900,
     background.integral.max = 1000,
     fit.method = "EXP",
     sequence.structure = c("TL", "pseudoIRSL1", "pseudoIRSL2"),
     main = "Pseudo pIRIR data set based on quartz OSL",
     plot.single = TRUE)


##(3) Perform pIRIR analysis (for this example with quartz OSL data!)
## Alternative for PDF output, uncomment and complete for usage
\dontrun{
tempfile <- tempfile(fileext = ".pdf")
pdf(file = tempfile, height = 15, width = 15)
  results <- analyse_pIRIRSequence(object,
         signal.integral.min = 1,
         signal.integral.max = 2,
         background.integral.min = 900,
         background.integral.max = 1000,
         fit.method = "EXP",
         main = "Pseudo pIRIR data set based on quartz OSL")

  dev.off()
}

} 

\section{How to cite}{
Kreutzer, S., 2019. analyse_pIRIRSequence(): Analyse post-IR IRSL measurement sequences. Function version 0.2.4. In: Kreutzer, S., Burow, C., Dietze, M., Fuchs, M.C., Schmidt, C., Fischer, M., Friedrich, J., 2019. Luminescence: Comprehensive Luminescence Dating Data AnalysisR package version 0.9.5. https://CRAN.R-project.org/package=Luminescence
}

\references{
Murray, A.S., Wintle, A.G., 2000. Luminescence dating of quartz
using an improved single-aliquot regenerative-dose protocol. Radiation
Measurements 32, 57-73. \doi{10.1016/S1350-4487(99)00253-X}

Thomsen, K.J., Murray, A.S., Jain, M., Boetter-Jensen, L., 2008. Laboratory
fading rates of various luminescence signals from feldspar-rich sediment
extracts. Radiation Measurements 43, 1474-1486.
\doi{10.1016/j.radmeas.2008.06.002}
}
\seealso{
\link{analyse_SAR.CWOSL}, \link{calc_OSLLxTxRatio}, \link{plot_GrowthCurve},
\linkS4class{RLum.Analysis}, \linkS4class{RLum.Results} \link{get_RLum}
}
\author{
Sebastian Kreutzer, IRAMAT-CRP2A, Universite Bordeaux Montaigne
(France)
, RLum Developer Team}
\keyword{datagen}
\keyword{plot}
