% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Real-Options.R
\name{LSM.RealOption}
\alias{LSM.RealOption}
\title{Value Capital Investment Projects Through Least-Squares Monte Carlo (LSM) Simulation:}
\usage{
LSM.RealOption(
  state.variables,
  NCF,
  CAPEX,
  dt,
  rf,
  construction = 0,
  orthogonal = "Laguerre",
  degree = 9,
  cross.product = TRUE,
  verbose = FALSE,
  debugging = FALSE
)
}
\arguments{
\item{state.variables}{\code{matrix} or \code{array}. The simulated states of the underlying stochastic variables. The first dimension corresponds to the simulated values
of the state variables at each discrete observation point. The second dimension corresponds to each individual simulation of the state variable. The third dimension
corresponds to each state variable considered.}

\item{NCF}{The net cash flows resulting from operating the underlying capital investment project for one discrete time step at the current simulated values of the state variables.
Each column corresponds to a simulated price path of underlying stochastic variables, and each row the net cash flows at a discrete time point for each simulated path.
The dimensions of object 'NCF' must match the first two dimensions of the object passed to the 'state.variables' argument.}

\item{CAPEX}{\code{numeric} or \code{vector} object. The initial capital investment of the project. This value can be either constant or time dependent. When the 'CAPEX' argument
is time dependent, it must be a vector object of length equal to the number of discrete observations of the simulations (i.e. the number of rows of both 'state.variables' and 'NCF')}

\item{dt}{Constant, discrete time step of simulated observations}

\item{rf}{The annual risk-free interest rate}

\item{construction}{An \code{integer} corresponding to the number of periods of construction of the underlying asset. The construction time represents the time between
the initial capital expenditure and when net cash flows are accrued, representing the construction time required of the investment project.}

\item{orthogonal}{\code{character}. The orthogonal polynomial used to develop basis functions that estimate the continuation value in the LSM simulation method.
\code{orthogonal} arguments available are: "Power", "Laguerre", "Jacobi", "Legendre", "Chebyshev", "Hermite". See \bold{details}.}

\item{degree}{The number of orthogonal polynomials used in the least-squares fit. See \bold{details}.}

\item{cross.product}{\code{logical}. Should a cross product of state variables be considered? Relevant only when the number of state variables
is greater than one}

\item{verbose}{\code{logical}. Should additional information be output? See \bold{values}.}

\item{debugging}{\code{logical} Should additional simulation information be output? See \bold{values}.}
}
\value{
The \code{LSM.RealOption} function returns a \code{list} object. The number of objects returned in the list is dependent upon the \code{logical}
values of arguments \code{verbose} and \code{debugging}.

\code{LSM.RealOption} by default returns 3 objects within the \code{list} class object:

\tabular{ll}{
\code{ROV} \tab 'Real Option value': The value of the capital investment project considering flexibility in the timing of investment.  \cr
\code{NPV} \tab 'Net Present Value': The value of the capital investment project considering immediate investment. \cr
\code{WOV} \tab 'Waiting Option Value': The value of the option to delay initial investment, equal to the difference between the ROV and NPV. \cr
}

When \code{verbose = T}, an additional 9 objects are returned within the \code{list} class object, providing further analysis into the capital investment project:

\tabular{ll}{
\code{ROV SE} \tab The standard error of 'ROV'. \cr
\code{NPV SE} \tab The standard error of 'NPV'. \cr
\code{WOV SE} \tab The standard error of 'WOV'. \cr
\code{Expected Timing} \tab The expected timing of investment, in years. \cr
\code{Expected Timing SE} \tab The standard error of the expected timing of investment. \cr
\code{Investment Prob} \tab The probability of investment within the forecasting horizon. \cr
\code{Cumulative Investment Prob} \tab The cumulative probability of investment at each discrete time point over the forecasting horizon. \cr
\code{PB} \tab The expected payback time of initial capital investment, in years. \cr
\code{PB SE} \tab The standard error of the expected payback time. \cr
}

When \code{debugging = T}, an additional 4 objects are returned within the \code{list} class object.
These objects provide information about the values of individual simulated price paths:

\tabular{ll}{
\code{Investment Period} \tab The time of investment of invested price paths. Price paths that did not trigger investment are represented as \code{NA} \cr
\code{Project Value} \tab The calculated project value at time zero for each simulated price path. The 'ROV' is equal to the mean of this vector. \cr
\code{Immediate Profit} \tab The profit resulting from immediate investment for each discrete time period and for all simulated price paths  \cr
\code{Running Present Value} \tab  The present value of all future cash flows of an investment project for each discrete time period and for all simulated price paths\cr
}
}
\description{
Given a set of state variables and associated net cash flows for an investment project simulated through Monte Carlo simulation,
solve for the real option value of a capital investment project through the least-squares Monte Carlo simulation method.
}
\details{
The \code{LSM.RealOption} function provides an implementation of the least-squares Monte Carlo (LSM) simulation approach to numerically approximate
the value of capital investment projects considering the flexibility of timing of investment under stochastically evolving uncertainty. The function provides flexibility in the stochastic behavior of the underlying uncertainty, with simulated values
of state variables provided within the \code{state.variables} argument. The \code{LSM.RealOption} function also provides analysis into the expected investment timing, probability, and the expected payback period of the project.

\bold{Least-Squares Monte Carlo Simulation:}

The least-squares Monte Carlo (LSM) simulation method is a numeric approach first presented by Longstaff and Schwartz (2001) that
approximates the value of options with early exercise opportunities. The LSM simulation method is considered one of the most efficient
methods of valuing American-style options due to its flexibility and computational efficiency. The approach can feature multiple
stochastically evolving underlying uncertainties, following both standard and exotic stochastic processes.

The LSM method first approximates stochastic variables through a stochastic process to develop cross-sectional information,
then directly estimates the continuation value of in-the-money simulation paths by "(regressing) the ex-post realized payoffs from
continuation on functions of the values of the state variables" (Longstaff and Schwartz, 2001).

\bold{Real Options Analysis}

Real options analysis of investment projects considers the value of the option to delay investment in a project until
underlying, stochastically evolving uncertainty has revealed itself. Real options analysis treats investment into capital investment projects as an optimal stopping problem, optimizing the timing of investment that maximizes the payoffs of investment under underlying stochastic uncertainty.
Real options analysis is also capable of evaluating the critical value of underlying state variables at which immediate investment into a project is optimal. See Dixit and Pindyck (1994) for more details of real options analysis.

The \code{LSM.RealOption} function considers the option to invest into a capital investment project within a finite forecasting horizon. Investment into the project results in accruing all future net cash flows (NCF) until the end of the forecasting horizon at the cost of the capital expenditure (CAPEX).
Immediate investment into a capital investment project is optimal when the waiting option value (WOV) is zero. Critical values of state variables at which immediate investment in optimal can therefore be obtained through finding the root of the WOV.

The primary difference between the \code{LSM.RealOption} and \code{LSM.AmericanOption} function is the way in which they evaluate the payoff of
exercise of the American-style option. The \code{LSM.AmericanOption} function considers the payoff of exercise to be a one time payoff (i.e. buying or selling the security in a vanilla call or put option) corresponding to the respective payoff argument.
The \code{LSM.RealOption} function instead, at each discrete time period, for each simulated price path, compares the sum of all remaining discounted cash flows that are accrued following immediate investment into
a project to the end of the forecasting horizon with the expected value of delaying investment. This is is known as the 'running present value' (RPV) of the project, which is the discretised present value of all
future cash flows of the project. The RPV of a project increases as the size of the discrete time step decreases, highlighting the need
for small discrete time steps to accurately value investment projects. This is due to the discounting effect of discounting larger
cash flows over greater time periods compared to smaller cash flows being discounted more frequently.

\bold{Orthogonal Polynomials:}

To improve the accuracy of estimation of continuation values, the economic values in each period are regressed on a linear
combination of a set of basis functions of the stochastic variables. These estimated regression parameters and the simulated
stochastic variables are then used to calculate the estimator for the expected economic values.

Longstaff and Schwartz (2001) state that as the conditional expectation of the continuation value belongs to a Hilbert space,
it can be represented by a combination of orthogonal basis functions. Increasing the number of stochastic state variables
therefore increases the number of required basis functions exponentially. The orthogonal polynomials available in the
\code{LSM.RealOptions} package are: Laguerre, Jacobi, Legendre (spherical), Hermite (probabilistic), Chebyshev (of the first kind).
The simple powers of state variables is further available. Explicit expressions of each of these orthogonal polynomials are
available within the textbook of Abramowitz and Stegun (1965).
}
\examples{
# Example: Value a capital investment project where the revenues follow a
# Geometric Brownian Motion stochastic process:

## Step 1 - Simulate asset prices:
AssetPrices <- GBM.Simulate(n = 100, t = 10, mu = 0.05,
                         sigma = 0.2, S0 = 100, dt = 1/2)

## Step 2 - Perform Real Option Analysis (ROA):
ROA <- LSM.RealOption(state.variables = AssetPrices,
                     NCF = AssetPrices - 100,
                     CAPEX = 1000,
                     dt = 1/2,
                     rf = 0.05)
}
\references{
Abramowitz, M., and I. A. Stegun, (1965). Handbook of mathematical functions with formulas, graphs, and mathematical tables. Courier Corporation.

Dixit, A. K., and R. S. Pindyck, (1994). Investment under uncertainty. Princeton university press.

Longstaff, F. A., and E. S. Schwartz, (2001). Valuing American options by simulation: a simple least-squares approach. The review of financial studies, 14(1), 113-147.
}
