.onLoad <- function(lib, pkg)
{
    ## Most of the holiday date series here have New Years Day for a few more
    ## years beyond the end of known data - having one day holiday in the
    ## data for a year helps catch lack of holiday data in other programs.
    ## These New Years Days holidays are calculated as the nearest weekday
    ## to Jan 1st and may not be NOT correct for the exchange, because
    ## different exchanges can use different rules for when to take
    ## the New Years Day holiday (if at all.)
    ##
    ## It is OK to have weekend dates in the holiday data -- will have no
    ## effect.

    ## Sources of trading holiday data:
    ##   http://www.jquantlib.org/index.php/Exchange_Holidays
    ##   http://www.asiaetrading.com/resource/holiday-calendar/

    ###########################################################################
    ##
    ## AUSEX
    ##
    ## 2004-5 cross checked against the exchange calendar:
    ## http://www.asx.com.au/about/Calendar_AA2.shtm
    ##
    registerHolidays('AUSEX', dateParse(
    c(
        "1990/01/01", "1990/01/29", "1990/04/13", "1990/04/16", "1990/04/25",
        "1990/06/11", "1990/12/25", "1990/12/26", "1990/12/31", "1991/01/01",
        "1991/01/28", "1991/03/29", "1991/04/01", "1991/04/25", "1991/06/10",
        "1991/12/25", "1991/12/26", "1992/01/01", "1992/01/27", "1992/04/17",
        "1992/04/20", "1992/06/08", "1992/12/25", "1992/12/26", "1992/12/28",
        "1993/01/01", "1993/04/09", "1993/04/12", "1993/06/14", "1993/12/27",
        "1993/12/28", "1994/01/03", "1994/01/26", "1994/04/01", "1994/04/04",
        "1994/04/25", "1994/06/13", "1994/10/03", "1994/12/26", "1994/12/27",
        "1995/01/02", "1995/01/26", "1995/04/14", "1995/04/17", "1995/04/25",
        "1995/06/12", "1995/10/02", "1995/12/25", "1995/12/26", "1996/01/01",
        "1996/01/02", "1996/01/26", "1996/04/05", "1996/04/08", "1996/04/25",
        "1996/06/10", "1996/10/07", "1996/12/25", "1996/12/26", "1997/01/01",
        "1997/01/27", "1997/03/28", "1997/03/31", "1997/04/25", "1997/06/09",
        "1997/08/04", "1997/10/06", "1997/12/25", "1997/12/26",

        "1998/01/01", "1998/01/26", "1998/04/10", "1998/04/13", "1998/06/08",
        "1998/12/25", "1998/12/28",

        "1999/01/01", "1999/01/26", "1999/04/02", "1999/04/05", "1999/06/14",
        "1999/12/27", "1999/12/28", "1999/12/31",

        "2000/01/03", "2000/01/26", "2000/04/21", "2000/04/24", "2000/04/25",
        "2000/06/12", "2000/12/25", "2000/12/26",

        "2001/01/01", "2001/01/26", "2001/04/13", "2001/04/16", "2001/04/25",
        "2001/06/11", "2001/12/25", "2001/12/26",

        "2002/01/01", "2002/01/28", "2002/03/29", "2002/04/01", "2002/04/25",
        "2002/06/10", "2002/12/25", "2002/12/26",

        "2003/01/01", "2003/01/27", "2003/04/18", "2003/04/21", "2003/04/25",
        "2003/06/09", "2003/12/25", "2003/12/26",

        "2004/01/01", "2004/01/26", "2004/04/09", "2004/04/12", "2004/06/14",
        "2004/12/25", "2004/12/27", "2004/12/28",

        "2005/01/03", "2005/01/26", "2005/03/25", "2005/03/28", "2005/04/25",
        "2005/06/13", "2005/12/26", "2005/12/27",

        ## from http://www.asx.com.au/supervision/operational/trading_calendar/2006.htm
        ## deleted quite a few days that were here -- those days were listed
        ## as "no settlement", but they are trading days.
        "2006/01/02", "2006/01/26", "2006/04/14", "2006/04/17",
        "2006/04/25", "2006/06/12", "2006/12/25", "2006/12/26",

        ## from http://www.asx.com.au/about/operational/trading_calendar/2007.htm
        ## deleted quite a few days that were here -- those days were listed
        ## as "no settlement", but they are trading days.
        "2007/01/01", "2007/01/26", "2007/04/06", "2007/04/09",
        "2007/04/25", "2007/06/11", "2007/12/25", "2007/12/26",

        ## from http://www.asx.com.au/about/operational/trading_calendar/asx/2008.htm
        "2008/01/01", "2008/01/28", "2008/03/21", "2008/03/24",
        "2008/04/25", "2008/06/09", "2008/12/25", "2008/12/26",

        ## from http://www.asx.com.au/about/operational/trading_calendar/asx/2009.htm
        "2009/01/01", "2009/01/26", "2009/04/10", "2009/04/13",
        "2009/06/08", "2009/12/25", "2009/12/28",

        ## from http://www.asx.com.au/trading_services/sycom-trading-calendar-2010.htm
        "2010/01/01", "2010/01/26", "2010/04/02", "2010/04/05", "2010/04/26",
        "2010/06/13", "2010/12/26", "2010/12/27",

        ## from http://www.asx.com.au/trading_services/asx-24-trading-calendar-2011.htm
        "2011/01/03", "2011/01/26", "2011/04/22", "2011/04/25", "2011/04/26",
        "2011/06/14", "2011/12/27", "2011/12/28",

        ## from http://www.asx.com.au/trading_services/asx-24-trading-calendar-2011.htm
        "2012/01/02", "2012/01/26", "2012/04/06", "2012/04/09", "2012/04/25",
        "2012/06/11", "2012/12/25", "2012/12/26",

        ## Check holidays on Bloomberg with "CDR AU 2009", though that shows up lots
        ## of extras that are holidays in Australia, but for which the market is open

        ## New Years Day for a few more years - having one day holiday in the
        ## data for a year helps catch lack of holiday data here and in the DB.
        ## These New Years Days holidays are probably NOT correct for the exchange,
        ## they are calculated using the rule:
        ## holiday.nearest.weekday(holiday.NewYears(years))
        ## However, different exchanges have different days as New Years Day holiday
        "2013/01/01", "2014/01/01", "2015/01/01"
    )))

    ###########################################################################
    ##
    ## GBREX
    ##
    registerHolidays('GBREX', dateParse(
    c(
        "1994/01/03", "1994/04/01", "1994/04/04", "1994/05/02", "1994/05/30", "1994/08/29",
        "1994/12/26", "1994/12/27", "1995/01/02", "1995/04/14", "1995/04/17", "1995/05/01",
        "1995/05/29", "1995/08/28", "1995/12/25", "1995/12/26", "1996/01/01", "1996/04/05",
        "1996/04/08", "1996/05/06", "1996/05/27", "1996/08/26", "1996/12/25", "1996/12/26",
        "1997/01/01", "1997/03/28", "1997/03/31", "1997/05/05", "1997/05/26", "1997/08/25",
        "1997/12/25", "1997/12/26", "1998/01/01", "1998/04/10", "1998/04/13", "1998/05/04",
        "1998/05/25", "1998/08/31", "1998/12/25", "1998/12/28", "1998/12/31", "1999/01/01",
        "1999/04/02", "1999/04/05", "1999/05/03", "1999/05/31", "1999/08/30", "1999/12/27",
        "1999/12/28", "1999/12/31", "2000/01/03", "2000/04/21", "2000/04/24", "2000/05/01",
        "2000/05/29", "2000/08/28", "2000/12/25", "2000/12/26", "2001/01/01", "2001/04/13",
        "2001/04/16", "2001/05/07", "2001/05/28", "2001/08/27", "2001/12/25", "2001/12/26",
        "2002/01/01", "2002/03/29", "2002/04/01", "2002/05/06", "2002/06/03",
        "2002/06/04", "2002/08/26", "2002/12/25", "2002/12/26", "2003/01/01", "2003/04/18",
        "2003/04/21", "2003/05/05", "2003/05/26", "2003/08/25", "2003/12/25", "2003/12/26",
        "2004/01/01", "2004/04/09", "2004/04/12", "2004/05/03", "2004/05/31", "2004/08/30",
        "2004/12/27", "2004/12/28",

        ## 2005 data is from http://www.londonstockexchange.com/en-gb/about/cooverview/
        ## businessdays.htm
        "2005/01/03", "2005/03/25", "2005/03/28", "2005/05/02", "2005/05/30", "2005/08/29",
        "2005/12/26", "2005/12/27",

        ## 2006 data is from http://www.londonstockexchange.com/en-gb/about/cooverview/
        ## businessdays/businessdays.htm
        "2006/01/02", "2006/04/14", "2006/04/17", "2006/05/01", "2006/05/29", "2006/08/28",
        "2006/12/25", "2006/12/26",

        ## 2007 data is from http://www.londonstockexchange.com/en-gb/about/cooverview/
        ## businessdays/businessdays.htm
        "2007/01/01", "2007/04/06", "2007/04/09", "2007/05/07", "2007/05/28", "2007/08/27",
        "2007/12/25", "2007/12/26",

        ## 2008 data is from http://www.londonstockexchange.com/en-gb/about/cooverview/
        ## businessdays/businessdays.htm
        "2008/01/01", "2008/03/21", "2008/03/24", "2008/05/05", "2008/05/26", "2008/08/25",
        "2008/12/25", "2008/12/26",

        ## 2009 data is from http://www.londonstockexchange.com/en-gb/about/cooverview/
        ## businessdays/businessdays.htm
        "2009/01/01", "2009/04/10", "2009/04/13", "2009/05/04", "2009/05/25", "2009/08/31",
        "2009/12/25", "2009/12/28",

        ## Check holidays on Bloomberg with "CDR GB 2009"

        ## New Years Day for a few more years - having one day holiday in the
        ## data for a year helps catch lack of holiday data here and in the DB.
        ## These New Years Days holidays are probably NOT correct for the exchange,
        ## they are calculated using the rule:
        ## holiday.nearest.weekday(holiday.NewYears(years))
        ## However, different exchanges have different days as New Years Day holiday
        "2010/01/01", "2010/12/31", "2012/01/02", "2013/01/01", "2014/01/01"
    )))

    ###########################################################################
    ##
    ## HKGEX
    ##
    registerHolidays('HKGEX', dateParse(
    c(
        "1996/01/01", "1996/02/19", "1996/02/20", "1996/02/21", "1996/04/04",
        "1996/04/05", "1996/04/08", "1996/06/17", "1996/06/20", "1996/08/26",
        "1996/10/21", "1996/12/25", "1996/12/26",

        "1997/01/01", "1997/02/06", "1997/02/07", "1997/03/28", "1997/03/31",
        "1997/06/09", "1997/06/30", "1997/07/01", "1997/07/02", "1997/08/18",
        "1997/09/17", "1997/10/01", "1997/10/02", "1997/10/10", "1997/12/25",
        "1997/12/26",

        "1998/01/01", "1998/01/28", "1998/01/29", "1998/01/30", "1998/04/06",
        "1998/04/10", "1998/04/13", "1998/07/01", "1998/08/17", "1998/10/01",
        "1998/10/02", "1998/10/06", "1998/10/28", "1998/12/25",

        "1999/01/01", "1999/02/16", "1999/02/17", "1999/02/18", "1999/04/02",
        "1999/04/05", "1999/04/06", "1999/06/18", "1999/07/01", "1999/09/16",
        "1999/10/01", "1999/10/18", "1999/12/27", "1999/12/31",

        "2000/02/04", "2000/02/07", "2000/04/04", "2000/04/21", "2000/04/24",
        "2000/05/01", "2000/05/11", "2000/06/06", "2000/09/13", "2000/10/02",
        "2000/10/06", "2000/12/25", "2000/12/26",

        "2001/01/01", "2001/01/24", "2001/01/25", "2001/01/26", "2001/04/05",
        "2001/04/13", "2001/04/16", "2001/04/30", "2001/05/01", "2001/06/25",
        "2001/07/02", "2001/07/06", "2001/07/25", "2001/10/01", "2001/10/02",
        "2001/10/25", "2001/12/25", "2001/12/26",

        "2002/01/01", "2002/02/12", "2002/02/13", "2002/02/14", "2002/03/29",
        "2002/04/01", "2002/04/05", "2002/05/01", "2002/05/20", "2002/07/01",
        "2002/10/01", "2002/10/14", "2002/12/25", "2002/12/26",

        "2003/01/01", "2003/01/31", "2003/02/03", "2003/04/18", "2003/04/21",
        "2003/05/01", "2003/05/08", "2003/06/04", "2003/07/01", "2003/09/12",
        "2003/10/01", "2003/12/25", "2003/12/26",

        "2004/01/01", "2004/01/22", "2004/01/23", "2004/04/05", "2004/04/09",
        "2004/04/12", "2004/05/26", "2004/06/22", "2004/07/01", "2004/09/29",
        "2004/10/01", "2004/10/22", "2004/12/27",

        "2005/02/09", "2005/02/10", "2005/02/11", "2005/03/25", "2005/03/28",
        "2005/04/05", "2005/05/02", "2005/05/16", "2005/07/01", "2005/09/19",
        "2005/10/11", "2005/12/26", "2005/12/27",

        ## from http://www.hkex.com.hk/tradinfo/tradcal/caln06e.htm
        "2006/01/02", "2006/01/30", "2006/01/31", "2006/04/05", "2006/04/14",
        "2006/04/17", "2006/05/01", "2006/05/05", "2006/05/31", "2006/10/02",
        "2006/10/30", "2006/12/25", "2006/12/26",

        ## from calendar db

        "2007/01/01", "2007/02/19", "2007/02/20", "2007/04/05", "2007/04/06",
        "2007/04/09", "2007/05/01", "2007/05/24", "2007/06/19", "2007/07/02",
        "2007/09/26", "2007/10/01", "2007/10/19", "2007/12/25", "2007/12/26",

        "2008/01/01", "2008/02/07", "2008/02/08", "2008/03/21", "2008/03/24",
        "2008/04/04", "2008/05/01", "2008/05/12", "2008/06/09", "2008/07/01",
        "2008/08/06", "2008/08/22", "2008/09/15", "2008/10/01", "2008/10/07",
        "2008/12/25", "2008/12/26",

        "2009/01/01", "2009/01/26", "2009/01/27", "2009/01/28", "2009/04/10",
        "2009/04/13", "2009/05/01", "2009/05/28", "2009/07/01", "2009/10/01",
        "2009/10/26", "2009/12/25",

        "2010/01/01", "2010/02/15", "2010/02/16", "2010/04/02", "2010/04/05",
        "2010/04/06", "2010/05/21", "2010/06/16", "2010/07/01", "2010/09/23",
        "2010/10/01", "2010/12/27",

        "2011/02/03", "2011/02/04", "2011/04/05", "2011/04/22", "2011/04/25",
        "2011/05/02", "2011/05/10", "2011/06/06", "2011/07/01", "2011/09/13",
        "2011/09/29", "2011/10/05", "2011/12/26", "2011/12/27",

        "2012/01/02", "2012/01/23", "2012/01/24", "2012/01/25", "2012/04/04",
        "2012/04/06", "2012/04/09", "2012/05/01", "2012/07/02", "2012/10/01",
        "2012/10/02", "2012/10/23", "2012/12/25", "2012/12/26",

        "2013/01/01", "2013/02/11", "2013/02/12", "2013/02/13", "2013/03/29",
        "2013/04/01", "2013/04/04", "2013/05/01", "2013/05/17", "2013/06/12",
        "2013/07/01", "2013/09/20", "2013/10/01", "2013/10/14", "2013/12/25",
        "2013/12/26",

        "2014/01/01", "2014/01/31", "2014/02/03", "2014/04/18", "2014/04/21",
        "2014/05/01", "2014/05/06", "2014/06/02", "2014/07/01", "2014/09/09",
        "2014/10/01", "2014/10/02", "2014/12/25", "2014/12/26",

        "2015/01/01", "2015/02/19", "2015/02/20", "2015/04/03", "2015/04/06",
        "2015/04/07", "2015/05/01", "2015/05/25", "2015/07/01", "2015/09/28",
        "2015/10/01", "2015/10/21", "2015/12/25",

        "2016/01/01", "2016/02/08", "2016/02/09", "2016/02/10", "2016/03/25",
        "2016/03/28", "2016/04/04", "2016/05/02", "2016/06/09", "2016/07/01",
        "2016/09/16", "2016/10/10", "2016/12/26", "2016/12/27",

        "2017/01/02", "2017/01/30", "2017/01/31", "2017/04/04", "2017/04/14",
        "2017/04/17", "2017/05/01", "2017/05/03", "2017/05/30", "2017/10/02",
        "2017/10/05", "2017/12/25", "2017/12/26",

        "2018/01/01", "2018/02/16", "2018/02/19", "2018/03/30", "2018/04/02",
        "2018/04/05", "2018/05/01", "2018/05/22", "2018/06/18", "2018/07/02",
        "2018/09/25", "2018/10/01", "2018/10/17", "2018/12/25", "2018/12/26",

        "2019/01/01", "2019/02/05", "2019/02/06", "2019/02/07", "2019/04/05",
        "2019/04/19", "2019/04/22", "2019/05/01", "2019/05/13", "2019/06/07",
        "2019/07/01", "2019/10/01", "2019/10/07", "2019/12/25", "2019/12/26",

        "2020/01/01", "2020/01/27", "2020/01/28", "2020/04/10", "2020/04/13",
        "2020/04/30", "2020/05/01", "2020/06/25", "2020/07/01", "2020/10/01",
        "2020/10/02", "2020/10/26", "2020/12/25",
        ## New Years for 2021.  Not published yet, so this is a guess.
        "2021/01/01"
    )))

    ###########################################################################
    ##
    ## JPNEX
    ##
    registerHolidays('JPNEX', dateParse(
    c(
        "1994/01/03", "1994/02/11", "1994/03/21", "1994/04/29", "1994/05/04",
        "1994/05/05", "1994/09/15", "1994/09/23", "1994/10/10", "1994/11/03", "1994/11/23",
        "1994/12/23", "1995/01/02", "1995/01/03", "1995/01/16", "1995/03/21", "1995/05/03",
        "1995/05/04", "1995/05/05", "1995/09/15", "1995/10/10", "1995/11/03", "1995/11/23",
        "1996/01/01", "1996/01/02", "1996/01/03", "1996/01/15", "1996/02/12", "1996/03/20",
        "1996/04/29", "1996/05/03", "1996/05/06", "1996/09/16", "1996/09/23", "1996/10/10",
        "1996/11/04", "1996/12/23", "1996/12/31", "1997/01/01", "1997/01/02", "1997/01/03",
        "1997/01/15", "1997/02/11", "1997/03/20", "1997/04/29", "1997/05/05", "1997/07/21",
        "1997/09/15", "1997/09/23", "1997/10/10", "1997/11/03", "1997/11/24", "1997/12/23",
        "1997/12/31", "1998/01/01", "1998/01/02", "1998/01/15", "1998/02/11", "1998/04/29",
        "1998/05/04", "1998/05/05", "1998/07/20", "1998/09/15",
        "1998/09/23", "1998/11/03", "1998/11/23", "1998/12/23", "1998/12/31", "1999/01/01",
        "1999/01/15", "1999/02/11", "1999/03/22", "1999/04/29", "1999/05/03", "1999/05/04",
        "1999/05/05", "1999/07/20", "1999/09/15", "1999/09/23", "1999/10/11", "1999/11/03",
        "1999/11/23", "1999/12/23", "1999/12/31", "2000/01/03", "2000/01/10", "2000/02/11",
        "2000/03/20", "2000/05/03", "2000/05/04", "2000/05/05", "2000/07/20", "2000/09/15",
        "2000/10/09", "2000/11/03", "2000/11/23", "2001/01/01", "2001/01/02", "2001/01/03",
        "2001/01/08", "2001/02/12", "2001/03/20", "2001/04/30", "2001/05/03", "2001/05/04",
        "2001/07/20", "2001/09/24", "2001/10/08", "2001/11/23", "2001/12/24", "2001/12/31",
        "2002/01/01", "2002/01/02", "2002/01/03", "2002/01/14", "2002/02/11", "2002/03/21",
        "2002/04/29", "2002/05/03", "2002/05/06", "2002/09/16", "2002/09/23", "2002/10/14",
        "2002/11/04", "2002/12/23", "2002/12/31", "2003/01/01", "2003/01/02",
        "2003/01/03", "2003/01/13", "2003/02/11", "2003/03/21", "2003/04/29",
        "2003/05/05", "2003/07/21", "2003/09/15", "2003/09/23", "2003/10/13", "2003/11/03",
        "2003/11/24", "2003/12/23", "2003/12/31",
        "2004/01/01", "2004/01/02", "2004/01/12", "2004/02/11", "2004/04/29", "2004/05/03",
        "2004/05/04", "2004/05/05", "2004/07/19", "2004/09/20", "2004/09/23", "2004/10/11",
        "2004/11/03", "2004/11/23", "2004/12/23", "2004/12/31",
        "2005/01/03", "2005/01/10", "2005/02/11", "2005/03/21", "2005/04/29", "2005/05/03",
        "2005/05/04", "2005/05/05", "2005/07/18", "2005/09/19", "2005/09/23", "2005/10/10",
        "2005/11/03", "2005/11/23", "2005/12/23",

        ## Recent data is from http://www.tse.or.jp/english/guide/calendar.html
        ## Supposedly 2003/05/03 was a holiday, but I think we had close prices for that day

        ## 2006/2007 data is from http://www.tse.or.jp/english/guide/calendar.html
        ## Note that not all the holidays are in the table -- have to read the
        ## text above the table to find Jan 2nd & 3rd(!!!)

        "2006/01/02", "2006/01/03", "2006/01/09", "2006/03/21", "2006/05/03",
        "2006/05/04", "2006/05/05", "2006/07/17", "2006/09/18", "2006/09/23",
        "2006/10/09", "2006/11/03", "2006/11/23",

        "2007/01/01", "2007/01/02", "2007/01/03", "2007/01/08", "2007/02/12", "2007/03/21", "2007/04/30",
        "2007/05/03", "2007/05/04", "2007/05/05", "2007/07/16", "2007/09/17",
        "2007/09/24", "2007/10/08", "2007/11/03", "2007/11/23", "2007/12/24", "2007/12/31",

        ## 2008 data is from http://www.tse.or.jp/english/about/calendar.html
        ## (visit http://www.tse.or.jp/english, then "About TSE"->Calendar)
        ## Note that not all the holidays are in the table -- have to read the
        ## text above the table to find Jan 2nd & 3rd & Dec 31 (!!!)

        "2008/01/01", "2008/01/02", "2008/01/03", "2008/01/14", "2008/02/11", "2008/03/20", "2008/04/29",
        "2008/05/05", "2008/05/06", "2008/07/21", "2008/09/15",
        "2008/09/23", "2008/10/13", "2008/11/03", "2008/11/24", "2008/12/23", "2008/12/31",

        ## 2009 data is from http://www.tse.or.jp/english/about/calendar.html
        ## (visit http://www.tse.or.jp/english, then "About TSE"->Calendar)
        ## Note that not all the holidays are in the table -- have to read the
        ## text above the table to find Jan 2nd & 3rd & Dec 31 (!!!)
        ## Not sure about the status of 05/03 and 05/04 -- these were also listed
        ## as holidays in 2008, but we didn't have them in our database.

        "2009/01/01", "2009/01/02", "2009/01/03", "2009/01/12", "2009/02/11", "2009/03/20", "2009/04/29",
        "2009/05/03", "2009/05/04", "2009/05/05", "2009/05/06", "2009/07/20", "2009/09/21",
        "2009/09/22", "2009/09/23", "2009/10/12", "2009/11/03", "2009/11/23", "2009/12/23", "2009/12/31",

        ## 2010 data is from http://www.timeanddate.com/calendar/?year=2010&country=26
        ## agrees with http://www.asiaetrading.com/resource/holiday-calendar/
        "2010/01/01", "2010/01/02", "2010/01/03", "2010/01/11", "2010/02/11", "2010/03/22", "2010/04/29",
        "2010/05/03", "2010/05/04", "2010/05/05", "2010/07/19", "2010/09/20",
        "2010/09/23", "2010/10/11", "2010/11/03", "2010/11/23", "2010/12/23", "2010/12/31",

        ## 2011 data is from http://www.tse.or.jp/english/about/calendar.html
        "2011/01/01", "2011/01/02", "2011/01/03", "2011/01/10", "2011/02/11", "2011/03/21", "2011/04/29",
        "2011/05/03", "2011/05/04", "2011/05/05", "2011/07/18", "2011/09/19",
        "2011/09/23", "2011/10/10", "2011/11/03", "2011/11/23", "2011/12/23", "2011/12/31",

        ## 2012 data is from http://www.tse.or.jp/english/about/calendar.html
        "2012/01/01", "2012/01/02", "2012/01/03", "2012/01/09", "2012/02/11", "2012/03/20", "2012/04/30",
        "2012/05/03", "2012/05/04", "2012/05/05", "2012/07/16", "2012/09/17",
        "2012/09/22", "2012/10/08", "2012/11/03", "2012/11/23", "2012/12/23", "2012/12/24", "2012/12/31",

        "2013/01/01", "2013/01/02", "2013/01/03", "2013/01/14", "2013/02/11",
        "2013/03/20", "2013/04/29", "2013/05/03", "2013/05/06", "2013/07/15",
        "2013/09/16", "2013/09/23", "2013/10/14", "2013/11/04", "2013/12/23",
        "2013/12/31",
        "2014/01/01", "2014/01/02", "2014/01/03", "2014/01/13", "2014/02/11",
        "2014/03/21", "2014/04/29", "2014/05/05", "2014/05/06", "2014/07/21",
        "2014/09/15", "2014/09/23", "2014/10/13", "2014/11/03", "2014/11/24",
        "2014/12/23", "2014/12/31",
        "2015/01/01", "2015/01/02", "2015/01/12", "2015/02/11", "2015/04/29",
        "2015/05/04", "2015/05/05", "2015/05/06", "2015/07/20", "2015/09/21",
        "2015/09/22", "2015/09/23", "2015/10/12", "2015/11/03", "2015/11/23",
        "2015/12/23", "2015/12/31",
        "2016/01/01", "2016/01/11", "2016/02/11", "2016/03/21", "2016/04/29",
        "2016/05/03", "2016/05/04", "2016/05/05", "2016/07/18", "2016/09/19",
        "2016/09/22", "2016/10/10", "2016/11/03", "2016/11/23", "2016/12/23",
        "2017/01/02", "2017/01/03", "2017/01/09", "2017/03/20", "2017/05/03",
        "2017/05/04", "2017/05/05", "2017/07/17", "2017/09/18", "2017/10/09",
        "2017/11/03", "2017/11/23",
        "2018/01/01", "2018/01/02", "2018/01/03", "2018/01/08", "2018/02/12",
        "2018/03/21", "2018/04/30", "2018/05/03", "2018/05/04", "2018/07/16",
        "2018/09/17", "2018/09/24", "2018/10/08", "2018/11/23", "2018/12/24",
        "2018/12/31",
        "2019/01/01", "2019/01/02", "2019/01/03", "2019/01/14", "2019/02/11",
        "2019/03/21", "2019/04/29", "2019/05/03", "2019/05/06", "2019/07/15",
        "2019/09/16", "2019/09/23", "2019/10/14", "2019/11/04", "2019/12/23",
        "2019/12/31",
        "2020/01/01", "2020/01/02", "2020/01/03", "2020/01/13", "2020/02/11",
        "2020/03/20", "2020/04/29", "2020/05/04", "2020/05/05", "2020/05/06",
        "2020/07/20", "2020/09/21", "2020/09/22", "2020/10/12", "2020/11/03",
        "2020/11/23", "2020/12/23", "2020/12/31",

        ## Check holidays on Bloomberg with "CDR JN 2009"

        ## New Years Day for a few more years - having one day holiday in the
        ## data for a year helps catch lack of holiday data here and in the DB.
        ## These New Years Days holidays are probably NOT correct for the exchange,
        ## they are calculated using the rule:
        ## holiday.nearest.weekday(holiday.NewYears(years))
        ## However, different exchanges have different days as New Years Day holiday
        "2021/01/01", "2022/01/01", "2023/01/01"
    )))

    ###########################################################################
    ##
    ## JPNEX.halfday (not up to date)
    ##
    registerHolidays('JPNEX.halfday', dateParse(
    c(
         "2003/01/06", "2003/12/30",
         "2004/01/05", "2004/12/30",
         "2005/01/04", "2005/12/30",
         "2006/01/04", "2006/12/29",
         "2006/01/04", "2006/12/28"
    )))


    ###########################################################################
    ##
    ## KOREX (not up to date)
    ##
    ## Data cross checked against the Bloomberg "CDR SK 2004"
    ##
    registerHolidays('KOREX', dateParse(
    c(
        "1996/01/01", "1996/01/02", "1996/02/19", "1996/02/20", "1996/03/01",
        "1996/04/05", "1996/04/11", "1996/05/01", "1996/05/05", "1996/05/24",
        "1996/06/04", "1996/07/17", "1996/08/15", "1996/09/26", "1996/09/27",
        "1996/09/28", "1996/10/03", "1996/12/25", "1996/12/28", "1996/12/30",
        "1996/12/31",

        "1997/01/02", "1997/02/07", "1997/02/08", "1997/03/01", "1997/04/05",
        "1997/05/01", "1997/05/05", "1997/05/24", "1997/07/17", "1997/08/15",
        "1997/09/15", "1997/09/16", "1997/09/17", "1997/10/03", "1997/12/18",
        "1997/12/25", "1997/12/29", "1997/12/30", "1997/12/31",

        "1998/01/01", "1998/01/02", "1998/01/27", "1998/01/28", "1998/01/29",
        "1998/05/01", "1998/05/05", "1998/06/04", "1998/06/06", "1998/07/17",
        "1998/08/15", "1998/10/03", "1998/10/05", "1998/10/06", "1998/12/12",
        "1998/12/19", "1998/12/25", "1998/12/26", "1998/12/29", "1998/12/30",
        "1998/12/31",

        "1999/01/01", "1999/01/02", "1999/02/15", "1999/02/16", "1999/02/17",
        "1999/03/01", "1999/04/05", "1999/05/01", "1999/05/05", "1999/05/22",
        "1999/06/06", "1999/07/17", "1999/08/15", "1999/09/23", "1999/09/24",
        "1999/09/25", "1999/10/03", "1999/12/25", "1999/12/29", "1999/12/30",
        "1999/12/31",

        "2000/01/01", "2000/01/02", "2000/01/03", "2000/02/04", "2000/02/05",
        "2000/02/06", "2000/03/01", "2000/04/05", "2000/04/13", "2000/05/01",
        "2000/05/05", "2000/05/11", "2000/06/06", "2000/07/17", "2000/08/15",
        "2000/09/11", "2000/09/12", "2000/09/13", "2000/10/03", "2000/12/25",
        "2000/12/27", "2000/12/28", "2000/12/29", "2000/12/30",

        "2001/01/01", "2001/01/23", "2001/01/24", "2001/01/25", "2001/03/01",
        "2001/04/05", "2001/05/01", "2001/05/05", "2001/06/06", "2001/07/17",
        "2001/08/15", "2001/10/01", "2001/10/02", "2001/10/03", "2001/12/25",
        "2001/12/31",

        "2002/01/01", "2002/02/11", "2002/02/12", "2002/02/13", "2002/03/01",
        "2002/04/05", "2002/05/01", "2002/05/05", "2002/05/19", "2002/06/06",
        "2002/06/13", "2002/07/01", "2002/07/17", "2002/08/15", "2002/09/20",
        "2002/09/21", "2002/10/03", "2002/12/19", "2002/12/25", "2002/12/31",

        "2003/01/01", "2003/01/31", "2003/02/01", "2003/03/01", "2003/04/05",
        "2003/05/01", "2003/05/05", "2003/05/08", "2003/06/06", "2003/07/17",
        "2003/08/15", "2003/09/10", "2003/09/11", "2003/09/12", "2003/10/03",
        "2003/12/25","2003/12/31",

        "2004/01/01", "2004/01/21", "2004/01/22", "2004/01/23", "2004/03/01",
        "2004/04/05", "2004/04/15", "2004/05/01", "2004/05/05", "2004/05/26",
        "2004/06/06", "2004/07/17", "2004/08/15", "2004/09/27", "2004/09/28",
        "2004/09/29", "2004/10/03", "2004/12/25", "2004/12/31",

        "2005/02/08", "2005/02/09", "2005/02/10", "2005/03/01", "2005/04/05",
        "2005/05/05", "2005/06/06", "2005/08/15", "2005/09/19", "2005/10/03",
        "2005/12/30",

        ## from http://sm.krx.co.kr/webeng/new_index.jsp (click on "Trading
        ## Hours & Calendar" under "One click service" -- had to view in
        ## internet explorer -- didn't see any 2006 holidays there)
        ## From the Bloomberg "CDR SK 2006"

        "2006/01/30", "2006/03/01", "2006/05/01", "2006/05/05", "2006/06/06",
        "2006/07/17", "2006/08/15", "2006/10/03", "2006/10/05", "2006/10/06",
        "2006/12/25",

        ## New Years for 2007.  Not published yet, so this is a guess.
        "2007/01/01"
    )))


    ###########################################################################
    ##
    ## NewYears
    ##
    registerHolidays('NewYears', as.Date(ISOdate(1980:2020, 1, 1)))

    ###########################################################################
    ##
    ## NEWYEAR
    ##
    ## The weekday day closest to new year's day, which seems to be the only day of
    ## the year on which there are no prices for international stocks.
    ## Note that this doesn't seem to be the holiday rule for GBR, at least: they
    ## have 01/03 as New Year's day for 2005 (2005/01/01 is a Saturday)
    ## Neither is holiday rule for NYSE: they don't have any day off
    ## for 2005/01/01.
    ##
    ##
    registerHolidays('NEWYEAR', dateParse(
    c(
        '1980/01/01','1981/01/01','1982/01/01','1982/12/31','1984/01/02','1985/01/01',
        '1986/01/01','1987/01/01','1988/01/01','1989/01/02','1990/01/01','1991/01/01',
        '1992/01/01','1993/01/01','1993/12/31','1995/01/02','1996/01/01','1997/01/01',
        '1998/01/01','1999/01/01','1999/12/31','2001/01/01','2002/01/01','2003/01/01',
        '2004/01/01','2004/12/31','2006/01/02','2007/01/01','2008/01/01','2009/01/01',
        '2010/01/01','2010/12/31','2012/01/02','2013/01/01','2014/01/01','2015/01/01',
        '2016/01/01','2017/01/02','2018/01/01','2019/01/01','2020/01/01'
    )))

    ###########################################################################
    ##
    ## NYSE
    ## This list includes holidays but not unsheduled closings, e.g., for funerals of
    ## presidents and for events that force closings.
    ##
    ##
    registerHolidays('NYSEPre70s', dateParse(
      c("1900/01/01", "1900/02/12", "1900/02/22", "1900/04/13", "1900/05/30", "1900/07/04",
        "1900/09/03", "1900/11/06", "1900/11/22", "1900/12/25", "1901/01/01", "1901/02/12",
        "1901/02/22", "1901/04/05", "1901/05/30", "1901/07/04", "1901/09/02", "1901/11/05",
        "1901/11/28", "1901/12/25", "1902/01/01", "1902/02/12", "1902/03/28", "1902/05/30",
        "1902/07/04", "1902/09/01", "1902/11/04", "1902/11/27", "1902/12/25", "1903/01/01",
        "1903/02/12", "1903/02/23", "1903/04/10", "1903/09/07", "1903/11/03", "1903/11/26",
        "1903/12/25", "1904/01/01", "1904/02/12", "1904/02/22", "1904/04/01", "1904/05/30",
        "1904/07/04", "1904/09/05", "1904/11/08", "1904/11/24", "1904/12/26", "1905/01/02",
        "1905/02/13", "1905/02/22", "1905/04/21", "1905/05/30", "1905/07/04", "1905/09/04",
        "1905/11/07", "1905/11/23", "1905/12/25", "1906/01/01", "1906/02/12", "1906/02/22",
        "1906/05/30", "1906/07/04", "1906/09/03", "1906/11/06", "1906/11/22", "1906/12/25",
        "1907/01/01", "1907/02/12", "1907/02/22", "1907/05/30", "1907/07/04", "1907/09/02",
        "1907/11/05", "1907/11/28", "1907/12/25", "1908/01/01", "1908/02/12", "1908/04/17",
        "1908/09/07", "1908/11/03", "1908/11/26", "1908/12/25", "1909/01/01", "1909/02/12",
        "1909/02/22", "1909/04/09", "1909/05/31", "1909/07/05", "1909/09/06", "1909/10/12",
        "1909/11/02", "1909/11/25", "1910/02/22", "1910/03/25", "1910/05/30", "1910/07/04",
        "1910/09/05", "1910/10/12", "1910/11/08", "1910/11/24", "1910/12/26", "1911/01/02",
        "1911/02/13", "1911/02/22", "1911/04/14", "1911/05/30", "1911/07/04", "1911/09/04",
        "1911/10/12", "1911/11/07", "1911/11/23", "1911/12/25", "1912/01/01", "1912/02/12",
        "1912/02/22", "1912/04/05", "1912/05/30", "1912/07/04", "1912/09/02", "1912/11/05",
        "1912/11/28", "1912/12/25", "1913/01/01", "1913/02/12", "1913/03/21", "1913/05/30",
        "1913/07/04", "1913/09/01", "1913/10/13", "1913/11/04", "1913/11/27", "1913/12/25",
        "1914/01/01", "1914/02/12", "1914/02/23", "1914/04/10", "1914/09/07", "1914/10/12",
        "1914/11/03", "1914/11/26", "1914/12/25", "1915/01/01", "1915/02/12", "1915/02/22",
        "1915/04/02", "1915/05/31", "1915/07/05", "1915/09/06", "1915/10/12", "1915/11/02",
        "1915/11/25", "1916/02/22", "1916/04/21", "1916/05/30", "1916/07/04", "1916/09/04",
        "1916/10/12", "1916/11/07", "1916/11/23", "1916/12/25", "1917/01/01", "1917/02/12",
        "1917/02/22", "1917/04/06", "1917/05/30", "1917/07/04", "1917/09/03", "1917/10/12",
        "1917/11/06", "1917/11/22", "1917/12/25", "1918/01/01", "1918/02/12", "1918/02/22",
        "1918/03/29", "1918/05/30", "1918/07/04", "1918/09/02", "1918/11/05", "1918/11/11",
        "1918/11/28", "1918/12/25", "1919/01/01", "1919/02/12", "1919/04/18", "1919/05/30",
        "1919/07/04", "1919/09/01", "1919/10/13", "1919/11/04", "1919/11/27", "1919/12/25",
        "1920/01/01", "1920/02/12", "1920/02/23", "1920/04/02", "1920/05/31", "1920/07/05",
        "1920/09/06", "1920/10/12", "1920/11/02", "1920/11/25", "1921/02/22", "1921/03/25",
        "1921/05/30", "1921/07/04", "1921/09/05", "1921/10/12", "1921/11/08", "1921/11/11",
        "1921/11/24", "1921/12/26", "1922/01/02", "1922/02/13", "1922/02/22", "1922/04/14",
        "1922/05/30", "1922/07/04", "1922/09/04", "1922/10/12", "1922/11/07", "1922/11/23",
        "1922/12/25", "1923/01/01", "1923/02/12", "1923/02/22", "1923/03/30", "1923/05/30",
        "1923/07/04", "1923/09/03", "1923/10/12", "1923/11/06", "1923/11/22", "1923/12/25",
        "1924/01/01", "1924/02/12", "1924/02/22", "1924/04/18", "1924/05/30", "1924/07/04",
        "1924/09/01", "1924/10/13", "1924/11/04", "1924/11/27", "1924/12/25", "1925/01/01",
        "1925/02/12", "1925/02/23", "1925/04/10", "1925/09/07", "1925/10/12", "1925/11/03",
        "1925/11/26", "1925/12/25", "1926/01/01", "1926/02/12", "1926/02/22", "1926/04/02",
        "1926/05/31", "1926/07/05", "1926/09/06", "1926/10/12", "1926/11/02", "1926/11/25",
        "1927/02/22", "1927/04/15", "1927/05/30", "1927/07/04", "1927/09/05", "1927/10/12",
        "1927/11/08", "1927/11/24", "1927/12/26", "1928/01/02", "1928/02/13", "1928/02/22",
        "1928/04/06", "1928/05/30", "1928/07/04", "1928/09/03", "1928/10/12", "1928/11/06",
        "1928/11/22", "1928/12/25", "1929/01/01", "1929/02/12", "1929/02/22", "1929/03/29",
        "1929/05/30", "1929/07/04", "1929/09/02", "1929/11/05", "1929/11/28", "1929/12/25",
        "1930/01/01", "1930/02/12", "1930/04/18", "1930/05/30", "1930/07/04", "1930/09/01",
        "1930/10/13", "1930/11/04", "1930/11/27", "1930/12/25", "1931/01/01", "1931/02/12",
        "1931/02/23", "1931/04/03", "1931/09/07", "1931/10/12", "1931/11/03", "1931/11/26",
        "1931/12/25", "1932/01/01", "1932/02/12", "1932/02/22", "1932/03/25", "1932/05/30",
        "1932/07/04", "1932/09/05", "1932/10/12", "1932/11/08", "1932/11/24", "1932/12/26",
        "1933/01/02", "1933/02/13", "1933/02/22", "1933/04/14", "1933/05/30", "1933/07/04",
        "1933/09/04", "1933/10/12", "1933/11/07", "1933/11/23", "1933/12/25", "1934/01/01",
        "1934/02/12", "1934/02/22", "1934/03/30", "1934/05/30", "1934/07/04", "1934/09/03",
        "1934/10/12", "1934/11/06", "1934/11/12", "1934/11/22", "1934/12/25", "1935/01/01",
        "1935/02/12", "1935/02/22", "1935/04/19", "1935/05/30", "1935/07/04", "1935/09/02",
        "1935/11/05", "1935/11/11", "1935/11/28", "1935/12/25", "1936/01/01", "1936/02/12",
        "1936/04/10", "1936/09/07", "1936/10/12", "1936/11/03", "1936/11/11", "1936/11/26",
        "1936/12/25", "1937/01/01", "1937/02/12", "1937/02/22", "1937/03/26", "1937/05/31",
        "1937/07/05", "1937/09/06", "1937/10/12", "1937/11/02", "1937/11/11", "1937/11/25",
        "1938/02/22", "1938/04/15", "1938/05/30", "1938/07/04", "1938/09/05", "1938/10/12",
        "1938/11/08", "1938/11/11", "1938/11/24", "1938/12/26", "1939/01/02", "1939/02/13",
        "1939/02/22", "1939/04/07", "1939/05/30", "1939/07/04", "1939/09/04", "1939/10/12",
        "1939/11/07", "1939/11/23", "1939/12/25", "1940/01/01", "1940/02/12", "1940/02/22",
        "1940/03/22", "1940/05/30", "1940/07/04", "1940/09/02", "1940/11/05", "1940/11/11",
        "1940/11/28", "1940/12/25", "1941/01/01", "1941/02/12", "1941/04/11", "1941/05/30",
        "1941/07/04", "1941/09/01", "1941/10/13", "1941/11/04", "1941/11/11", "1941/11/27",
        "1941/12/25", "1942/01/01", "1942/02/12", "1942/02/23", "1942/04/03", "1942/09/07",
        "1942/10/12", "1942/11/03", "1942/11/11", "1942/11/26", "1942/12/25", "1943/01/01",
        "1943/02/12", "1943/02/22", "1943/04/23", "1943/05/31", "1943/07/05", "1943/09/06",
        "1943/10/12", "1943/11/02", "1943/11/11", "1943/11/25", "1944/02/22", "1944/04/07",
        "1944/05/30", "1944/07/04", "1944/09/04", "1944/10/12", "1944/11/07", "1944/11/23",
        "1944/12/25", "1945/01/01", "1945/02/12", "1945/02/22", "1945/03/30", "1945/05/30",
        "1945/07/04", "1945/09/03", "1945/10/12", "1945/11/06", "1945/11/12", "1945/11/22",
        "1945/12/25", "1946/01/01", "1946/02/12", "1946/02/22", "1946/04/19", "1946/05/30",
        "1946/07/04", "1946/09/02", "1946/11/05", "1946/11/11", "1946/11/28", "1946/12/25",
        "1947/01/01", "1947/02/12", "1947/04/04", "1947/05/30", "1947/07/04", "1947/09/01",
        "1947/10/13", "1947/11/04", "1947/11/11", "1947/11/27", "1947/12/25", "1948/01/01",
        "1948/02/12", "1948/02/23", "1948/03/26", "1948/05/31", "1948/07/05", "1948/09/06",
        "1948/10/12", "1948/11/02", "1948/11/11", "1948/11/25", "1949/02/22", "1949/04/15",
        "1949/05/30", "1949/07/04", "1949/09/05", "1949/10/12", "1949/11/08", "1949/11/11",
        "1949/11/24", "1949/12/26", "1950/01/02", "1950/02/13", "1950/02/22", "1950/04/07",
        "1950/05/30", "1950/07/04", "1950/09/04", "1950/10/12", "1950/11/07", "1950/11/23",
        "1950/12/25", "1951/01/01", "1951/02/12", "1951/02/22", "1951/03/23", "1951/05/30",
        "1951/07/04", "1951/09/03", "1951/10/12", "1951/11/06", "1951/11/12", "1951/11/22",
        "1951/12/25", "1952/01/01", "1952/02/12", "1952/02/22", "1952/04/11", "1952/05/30",
        "1952/07/04", "1952/09/01", "1952/10/13", "1952/11/04", "1952/11/11", "1952/11/27",
        "1952/12/25", "1953/01/01", "1953/02/12", "1953/02/23", "1953/04/03", "1953/09/07",
        "1953/10/12", "1953/11/03", "1953/11/11", "1953/11/26", "1953/12/25", "1954/01/01",
        "1954/02/22", "1954/04/16", "1954/05/31", "1954/07/05", "1954/09/06", "1954/11/02",
        "1954/11/25", "1955/02/22", "1955/04/08", "1955/05/30", "1955/07/04", "1955/09/05",
        "1955/11/08", "1955/11/24", "1955/12/26", "1956/01/02", "1956/02/22", "1956/03/30",
        "1956/05/30", "1956/07/04", "1956/09/03", "1956/11/06", "1956/11/22", "1956/12/25",
        "1957/01/01", "1957/02/22", "1957/04/19", "1957/05/30", "1957/07/04", "1957/09/02",
        "1957/11/05", "1957/11/28", "1957/12/25", "1958/01/01", "1958/04/04", "1958/05/30",
        "1958/07/04", "1958/09/01", "1958/11/04", "1958/11/27", "1958/12/25", "1959/01/01",
        "1959/02/23", "1959/03/27", "1959/07/03", "1959/09/07", "1959/11/03", "1959/11/26",
        "1959/12/25", "1960/01/01", "1960/02/22", "1960/04/15", "1960/05/30", "1960/07/04",
        "1960/09/05", "1960/11/08", "1960/11/24", "1960/12/26", "1961/01/02", "1961/02/22",
        "1961/03/31", "1961/05/30", "1961/07/04", "1961/09/04", "1961/11/07", "1961/11/23",
        "1961/12/25", "1962/01/01", "1962/02/22", "1962/04/20", "1962/05/30", "1962/07/04",
        "1962/09/03", "1962/11/06", "1962/11/22", "1962/12/25", "1963/01/01", "1963/02/22",
        "1963/04/12", "1963/05/30", "1963/07/04", "1963/09/02", "1963/11/05", "1963/11/28",
        "1963/12/25", "1964/01/01", "1964/02/21", "1964/03/27", "1964/07/03", "1964/09/07",
        "1964/11/03", "1964/11/26", "1964/12/25", "1965/01/01", "1965/02/22", "1965/04/16",
        "1965/05/31", "1965/07/05", "1965/09/06", "1965/11/02", "1965/11/25", "1965/12/24",
        "1966/02/22", "1966/04/08", "1966/05/30", "1966/07/04", "1966/09/05", "1966/11/08",
        "1966/11/24", "1966/12/26", "1967/01/02", "1967/02/22", "1967/03/24", "1967/05/30",
        "1967/07/04", "1967/09/04", "1967/11/07", "1967/11/23", "1967/12/25", "1968/01/01",
        "1968/02/22", "1968/04/12", "1968/05/30", "1968/07/04", "1968/09/02", "1968/11/05",
        "1968/11/28", "1968/12/25", "1969/01/01", "1969/02/21", "1969/04/04", "1969/05/30",
        "1969/07/04", "1969/09/01", "1969/11/27", "1969/12/25", "1970/01/01", "1970/02/23",
        "1970/03/27", "1970/07/03", "1970/09/07", "1970/11/26", "1970/12/25", "1971/01/01",
        "1971/02/15", "1971/04/09", "1971/05/31", "1971/07/05", "1971/09/06", "1971/11/25",
        "1971/12/24", "1972/02/21", "1972/03/31", "1972/05/29", "1972/07/04", "1972/09/04",
        "1972/11/07", "1972/11/23", "1972/12/25", "1973/01/01", "1973/02/19", "1973/04/20",
        "1973/05/28", "1973/07/04", "1973/09/03", "1973/11/22", "1973/12/25", "1974/01/01",
        "1974/02/18", "1974/04/12", "1974/05/27", "1974/07/04", "1974/09/02", "1974/11/28",
        "1974/12/25", "1975/01/01", "1975/02/17", "1975/03/28", "1975/05/26", "1975/07/04",
        "1975/09/01", "1975/11/27", "1975/12/25", "1976/01/01", "1976/02/16", "1976/04/16",
        "1976/05/31", "1976/07/05", "1976/09/06", "1976/11/02", "1976/11/25", "1976/12/24",
        "1977/02/21", "1977/04/08", "1977/05/30", "1977/07/04", "1977/09/05", "1977/11/24",
        "1977/12/26", "1978/01/02", "1978/02/20", "1978/03/24", "1978/05/29", "1978/07/04",
        "1978/09/04", "1978/11/23", "1978/12/25", "1979/01/01", "1979/02/19", "1979/04/13",
        "1979/05/28", "1979/07/04", "1979/09/03", "1979/11/22", "1979/12/25")))

    registerHolidays('NYSE', dateParse(
    c('1970/01/01','1970/02/23','1970/03/27','1970/07/03','1970/09/07','1970/11/26',
        '1970/12/25','1971/01/01','1971/02/15','1971/04/09','1971/05/31','1971/07/05',
        '1971/09/06','1971/11/25','1971/12/24','1972/02/21','1972/03/31','1972/05/29',
        '1972/07/04','1972/09/04','1972/11/07','1972/11/23','1972/12/25','1973/01/01',
        '1973/02/19','1973/04/20','1973/05/28','1973/07/04','1973/09/03',
        '1973/11/22','1973/12/25','1974/01/01','1974/02/18','1974/04/12','1974/05/27',
        '1974/07/04','1974/09/02','1974/11/28','1974/12/25','1975/01/01','1975/02/17',
        '1975/03/28','1975/05/26','1975/07/04','1975/09/01','1975/11/27','1975/12/25',
        '1976/01/01','1976/02/16','1976/04/16','1976/05/31','1976/07/05','1976/09/06',
        '1976/11/02','1976/11/25','1976/12/24','1977/02/21','1977/04/08','1977/05/30',
        '1977/07/04','1977/09/05','1977/11/24','1977/12/26','1978/01/02',
        '1978/02/20','1978/03/24','1978/05/29','1978/07/04','1978/09/04','1978/11/23',
        '1978/12/25','1979/01/01','1979/02/19','1979/04/13','1979/05/28','1979/07/04',
        '1979/09/03','1979/11/22','1979/12/25',
        '1980/01/01','1980/02/18','1980/04/04','1980/05/26','1980/07/04','1980/09/01',
        '1980/11/04','1980/11/27','1980/12/25','1981/01/01','1981/02/16','1981/04/17',
        '1981/05/25','1981/07/03','1981/09/07','1981/11/26','1981/12/25','1982/01/01',
        '1982/02/15','1982/04/09','1982/05/31','1982/07/05','1982/09/06','1982/11/25',
        '1982/12/24','1983/02/21','1983/04/01','1983/05/30','1983/07/04','1983/09/05',
        '1983/11/24','1983/12/26','1984/01/02','1984/02/20','1984/04/20','1984/05/28',
        '1984/07/04','1984/09/03','1984/11/22','1984/12/25','1985/01/01','1985/02/18',
        '1985/04/05','1985/05/27','1985/07/04','1985/09/02','1985/11/28','1985/12/25',
        '1986/01/01','1986/02/17','1986/03/28','1986/05/26','1986/07/04','1986/09/01',
        '1986/11/27','1986/12/25','1987/01/01','1987/02/16','1987/04/17','1987/05/25',
        '1987/07/03','1987/09/07','1987/11/26','1987/12/25','1988/01/01','1988/02/15',
        '1988/04/01','1988/05/30','1988/07/04','1988/09/05','1988/11/24','1988/12/26',
        '1989/01/02','1989/02/20','1989/03/24','1989/05/29','1989/07/04','1989/09/04',
        '1989/11/23','1989/12/25','1990/01/01','1990/02/19','1990/04/13','1990/05/28',
        '1990/07/04','1990/09/03','1990/11/22','1990/12/25','1991/01/01','1991/02/18',
        '1991/03/29','1991/05/27','1991/07/04','1991/09/02','1991/11/28','1991/12/25',
        '1992/01/01','1992/02/17','1992/04/17','1992/05/25','1992/07/03','1992/09/07',
        '1992/11/26','1992/12/25','1993/01/01','1993/02/15','1993/04/09','1993/05/31',
        '1993/07/05','1993/09/06','1993/11/25','1993/12/24','1994/02/21','1994/04/01',
        '1994/05/30','1994/07/04','1994/09/05','1994/11/24','1994/12/26','1995/01/02',
        '1995/02/20','1995/04/14','1995/05/29','1995/07/04','1995/09/04','1995/11/23',
        '1995/12/25','1996/01/01','1996/02/19','1996/04/05','1996/05/27','1996/07/04',
        '1996/09/02','1996/11/28','1996/12/25','1997/01/01','1997/02/17','1997/03/28',
        '1997/05/26','1997/07/04','1997/09/01','1997/11/27','1997/12/25','1998/01/01',
        '1998/01/19','1998/02/16','1998/04/10','1998/05/25','1998/07/03','1998/09/07',
        '1998/11/26','1998/12/25','1999/01/01','1999/01/18','1999/02/15','1999/04/02',
        '1999/05/31','1999/07/05','1999/09/06','1999/11/25','1999/12/24','2000/01/17',
        '2000/02/21','2000/04/21','2000/05/29','2000/07/04','2000/09/04','2000/11/23',
        '2000/12/25','2001/01/01','2001/01/15','2001/02/19','2001/04/13','2001/05/28',
        '2001/07/04','2001/09/03','2001/11/22','2001/12/25','2002/01/01','2002/01/21',
        '2002/02/18','2002/03/29','2002/05/27','2002/07/04','2002/09/02','2002/11/28',
        '2002/12/25','2003/01/01','2003/01/20','2003/02/17','2003/04/18','2003/05/26',
        '2003/07/04','2003/09/01','2003/11/27','2003/12/25','2004/01/01','2004/01/19',
        '2004/02/16','2004/04/09','2004/05/31','2004/07/05','2004/09/06','2004/11/25',
        '2004/12/24','2005/01/17','2005/02/21','2005/03/25','2005/05/30','2005/07/04',
        '2005/09/05','2005/11/24','2005/12/26','2006/01/02','2006/01/16','2006/02/20',
        '2006/04/14','2006/05/29','2006/07/04','2006/09/04','2006/11/23','2006/12/25',
        '2007/01/01','2007/01/15','2007/02/19','2007/04/06','2007/05/28','2007/07/04',
        '2007/09/03','2007/11/22','2007/12/25','2008/01/01','2008/01/21','2008/02/18',
        '2008/03/21','2008/05/26','2008/07/04','2008/09/01','2008/11/27','2008/12/25',
        '2009/01/01','2009/01/19','2009/02/16','2009/04/10','2009/05/25','2009/07/03',
        '2009/09/07','2009/11/26','2009/12/25','2010/01/01','2010/01/18','2010/02/15',
        '2010/04/02','2010/05/31','2010/07/05','2010/09/06','2010/11/25','2010/12/24',
        '2011/01/17','2011/02/21','2011/04/22','2011/05/30','2011/07/04','2011/09/05',
        '2011/11/24','2011/12/26','2012/01/02','2012/01/16','2012/02/20','2012/04/06',
        '2012/05/28','2012/07/04','2012/09/03','2012/11/22','2012/12/25','2013/01/01',
        '2013/01/21','2013/02/18','2013/03/29','2013/05/27','2013/07/04','2013/09/02',
        '2013/11/28','2013/12/25','2014/01/01','2014/01/20','2014/02/17','2014/04/18',
        '2014/05/26','2014/07/04','2014/09/01','2014/11/27','2014/12/25','2015/01/01',
        '2015/01/19','2015/02/16','2015/04/03','2015/05/25','2015/07/03','2015/09/07',
        '2015/11/26','2015/12/25','2016/01/01','2016/01/18','2016/02/15','2016/03/25',
        '2016/05/30','2016/07/04','2016/09/05','2016/11/24','2016/12/26','2017/01/02',
        '2017/01/16','2017/02/20','2017/04/14','2017/05/29','2017/07/04','2017/09/04',
        '2017/11/23','2017/12/25','2018/01/01','2018/01/15','2018/02/19','2018/03/30',
        '2018/05/28','2018/07/04','2018/09/03','2018/11/22','2018/12/25','2019/01/01',
        '2019/01/21','2019/02/18','2019/04/19','2019/05/27','2019/07/04','2019/09/02',
        '2019/11/28','2019/12/25','2020/01/01','2020/01/20','2020/02/17','2020/04/10',
        '2020/05/25','2020/07/03','2020/09/07','2020/11/26','2020/12/25'
    )))

    ###########################################################################
    ##
    ## NYSE.Closings
    ## These need to be updated when there is a closing.
    ## NYSE traditionally closes for the funeral of a President of the US
    ##
    registerHolidays('NYSE.Closings', dateParse(
    c(
         "1973/01/25",  # funeral of former President Lyndon B. Johnson
         "1977/07/14",  # Closed due to blackout in New York
         "1994/04/27",  # funeral of Richard M. Nixon
         "2001/09/11",  # Attack on World Trade Center
         "2001/09/12",  # Attack on World Trade Center
         "2001/09/13",  # Attack on World Trade Center
         "2001/09/14",  # Attack on World Trade Center
         "2004/06/11",  # funeral of former President Ronald Reagan
         "2007/01/02",  # National Day of mouring for Gerald Ford
         "2012/10/29","2012/10/30"   # Hurricane Sandy

    )))

    ###########################################################################
    ##
    ## NYSEC
    ##
    ## Combine holidays and historical closings (e.g., funeral of Richard Nixon,
    ## World Trade Center attack).
    ##
    registerHolidays("NYSEC", c(holidays(1980:2020, "NYSE"),
                                holidays(1980:2020, "NYSE.Closings",
                                         silent = TRUE)))
    registerHolidays("NYSELong", c(holidays(1900:1969, "NYSEPre70s"),
                                   holidays(1970:2020, "NYSE"),
                                   holidays(1970:2020, "NYSE.Closings",
                                         silent = TRUE)))

    ###########################################################################
    ##
    ## NYSEC.halfday (not up to date)
    ##
    registerHolidays('NYSE.halfday', dateParse(
    c(
         "2003/07/03", "2003/11/28", "2003/12/24", "2003/12/26"
    )))

    ###########################################################################
    ##
    ## USFed United States federal holidays
    ##
    ## From a document originally at:
    ##   http://www.usis.usemb.se/Holidays/celebrate/intro.html
    ##   found at http://infousa.state.gov/life/symbceleb/docs/celebrat.pdf in Jan 2013
    ##
    ## Ten holidays per year are proclaimed by the federal government. They are as follows:
    ## New Year's Day (January 1 )
    ## Martin Luther King Day (traditional - January 15)
    ##                        (official - third Monday in January)
    ## Presidents' Day (traditional - February 12 & February 22)
    ##                 (official - third Monday in February)
    ## Memorial Day (traditional - May 30)
    ##              (official - last Monday in May)
    ## Independence Day (July 4)
    ## Labor Day (first Monday in September)
    ## Columbus Day (traditional - October 12)
    ##              (official - second Monday in October)
    ## Veterans' Day (traditional - November 11)
    ##               (official - second Monday in November)
    ## Thanksgiving Day (fourth Thursday in November)
    ## Christmas Day (December 25)
    ##

    ## In 1971, the dates of many federal holidays were
    ## officially moved to the nearest Monday by
    ## then-President Richard Nixon. There are four holidays
    ## which are not necessarily celebrated on Mondays:
    ## Thanksgiving Day, New Year's Day, Independence Day
    ## and Christmas Day. When New Year's Day, Independence
    ## Day, or Christmas Day falls on a Sunday, the next day
    ## is also a holiday.  When one of these holidays falls
    ## on a Saturday, the previous day is also a holiday.
    ##
    ## Federal government offices, including the post
    ## office, are always closed on all federal holidays.
    ## Schools and businesses close on major holidays like
    ## Independence Day and Christmas Day but may not always
    ## be closed, for example, on Presidents' Day or
    ## Veterans' Day.
    ##
    ##
    ## 2nd monday in October
    ## dateAlign(dateSeq(dateParse('1990-10-01'), len=5, by='years', align.by=F), week.align=1, dir=1, by='weeks')+7
    ## 2nd monday in November
    ## dateAlign(dateSeq(dateParse('1990-11-01'), len=5, by='years', align.by=F), week.align=1, dir=1, by='weeks')+7
    ##
    registerHolidays('USFed', dateParse(
      c("1971/01/01", "1971/01/18", "1971/02/15", "1971/05/31",
	"1971/07/05", "1971/09/06", "1971/10/11", "1971/11/11", "1971/11/25", "1971/12/24",
	"1971/12/31", "1972/01/17", "1972/02/21", "1972/05/29", "1972/07/04", "1972/09/04",
	"1972/10/09", "1972/11/10", "1972/11/23", "1972/12/25", "1973/01/01", "1973/01/15",
	"1973/02/19", "1973/05/28", "1973/07/04", "1973/09/03", "1973/10/08", "1973/11/12",
	"1973/11/22", "1973/12/25", "1974/01/01", "1974/01/21", "1974/02/18", "1974/05/27",
	"1974/07/04", "1974/09/02", "1974/10/14", "1974/11/11", "1974/11/28", "1974/12/25",
	"1975/01/01", "1975/01/20", "1975/02/17", "1975/05/26", "1975/07/04", "1975/09/01",
	"1975/10/13", "1975/11/11", "1975/11/27", "1975/12/25", "1976/01/01", "1976/01/19",
	"1976/02/16", "1976/05/31", "1976/07/05", "1976/09/06", "1976/10/11", "1976/11/11",
	"1976/11/25", "1976/12/24", "1976/12/31", "1977/01/17", "1977/02/21", "1977/05/30",
	"1977/07/04", "1977/09/05", "1977/10/10", "1977/11/11", "1977/11/24", "1977/12/26",
	"1978/01/02", "1978/01/16", "1978/02/20", "1978/05/29", "1978/07/04", "1978/09/04",
	"1978/10/09", "1978/11/10", "1978/11/23", "1978/12/25", "1979/01/01", "1979/01/15",
	"1979/02/19", "1979/05/28", "1979/07/04", "1979/09/03", "1979/10/08", "1979/11/12",
	"1979/11/22", "1979/12/25", "1980/01/01", "1980/01/21", "1980/02/18", "1980/05/26",
	"1980/07/04", "1980/09/01", "1980/10/13", "1980/11/11", "1980/11/27", "1980/12/25",
	"1981/01/01", "1981/01/19", "1981/02/16", "1981/05/25", "1981/07/03", "1981/09/07",
	"1981/10/12", "1981/11/11", "1981/11/26", "1981/12/25", "1982/01/01", "1982/01/18",
	"1982/02/15", "1982/05/31", "1982/07/05", "1982/09/06", "1982/10/11", "1982/11/11",
	"1982/11/25", "1982/12/24", "1982/12/31", "1983/01/17", "1983/02/21", "1983/05/30",
	"1983/07/04", "1983/09/05", "1983/10/10", "1983/11/11", "1983/11/24", "1983/12/26",
	"1984/01/02", "1984/01/16", "1984/02/20", "1984/05/28", "1984/07/04", "1984/09/03",
	"1984/10/08", "1984/11/12", "1984/11/22", "1984/12/25", "1985/01/01", "1985/01/21",
	"1985/02/18", "1985/05/27", "1985/07/04", "1985/09/02", "1985/10/14", "1985/11/11",
	"1985/11/28", "1985/12/25", "1986/01/01", "1986/01/20", "1986/02/17", "1986/05/26",
	"1986/07/04", "1986/09/01", "1986/10/13", "1986/11/11", "1986/11/27", "1986/12/25",
	"1987/01/01", "1987/01/19", "1987/02/16", "1987/05/25", "1987/07/03", "1987/09/07",
	"1987/10/12", "1987/11/11", "1987/11/26", "1987/12/25", "1988/01/01", "1988/01/18",
	"1988/02/15", "1988/05/30", "1988/07/04", "1988/09/05", "1988/10/10", "1988/11/11",
	"1988/11/24", "1988/12/26", "1989/01/02", "1989/01/16", "1989/02/20", "1989/05/29",
	"1989/07/04", "1989/09/04", "1989/10/09", "1989/11/10", "1989/11/23", "1989/12/25",
	"1990/01/01", "1990/01/15", "1990/02/19", "1990/05/28", "1990/07/04", "1990/09/03",
	"1990/10/08", "1990/11/12", "1990/11/22", "1990/12/25", "1991/01/01", "1991/01/21",
	"1991/02/18", "1991/05/27", "1991/07/04", "1991/09/02", "1991/10/14", "1991/11/11",
	"1991/11/28", "1991/12/25", "1992/01/01", "1992/01/20", "1992/02/17", "1992/05/25",
	"1992/07/03", "1992/09/07", "1992/10/12", "1992/11/11", "1992/11/26", "1992/12/25",
	"1993/01/01", "1993/01/18", "1993/02/15", "1993/05/31", "1993/07/05", "1993/09/06",
	"1993/10/11", "1993/11/11", "1993/11/25", "1993/12/24", "1993/12/31", "1994/01/17",
	"1994/02/21", "1994/05/30", "1994/07/04", "1994/09/05", "1994/10/10", "1994/11/11",
	"1994/11/24", "1994/12/26", "1995/01/02", "1995/01/16", "1995/02/20", "1995/05/29",
	"1995/07/04", "1995/09/04", "1995/10/09", "1995/11/10", "1995/11/23", "1995/12/25",
	"1996/01/01", "1996/01/15", "1996/02/19", "1996/05/27", "1996/07/04", "1996/09/02",
	"1996/10/14", "1996/11/11", "1996/11/28", "1996/12/25", "1997/01/01", "1997/01/20",
	"1997/02/17", "1997/05/26", "1997/07/04", "1997/09/01", "1997/10/13", "1997/11/11",
	"1997/11/27", "1997/12/25", "1998/01/01", "1998/01/19", "1998/02/16", "1998/05/25",
	"1998/07/03", "1998/09/07", "1998/10/12", "1998/11/11", "1998/11/26", "1998/12/25",
	"1999/01/01", "1999/01/18", "1999/02/15", "1999/05/31", "1999/07/05", "1999/09/06",
	"1999/10/11", "1999/11/11", "1999/11/25", "1999/12/24", "1999/12/31", "2000/01/17",
	"2000/02/21", "2000/05/29", "2000/07/04", "2000/09/04", "2000/10/09", "2000/11/10",
	"2000/11/23", "2000/12/25", "2001/01/01", "2001/01/15", "2001/02/19", "2001/05/28",
	"2001/07/04", "2001/09/03", "2001/10/08", "2001/11/12", "2001/11/22", "2001/12/25",
	"2002/01/01", "2002/01/21", "2002/02/18", "2002/05/27", "2002/07/04", "2002/09/02",
	"2002/10/14", "2002/11/11", "2002/11/28", "2002/12/25", "2003/01/01", "2003/01/20",
	"2003/02/17", "2003/05/26", "2003/07/04", "2003/09/01", "2003/10/13", "2003/11/11",
	"2003/11/27", "2003/12/25", "2004/01/01", "2004/01/19", "2004/02/16", "2004/05/31",
	"2004/07/05", "2004/09/06", "2004/10/11", "2004/11/11", "2004/11/25", "2004/12/24",
	"2004/12/31", "2005/01/17", "2005/02/21", "2005/05/30", "2005/07/04", "2005/09/05",
	"2005/10/10", "2005/11/11", "2005/11/24", "2005/12/26", "2006/01/02", "2006/01/16",
	"2006/02/20", "2006/05/29", "2006/07/04", "2006/09/04", "2006/10/09", "2006/11/10",
	"2006/11/23", "2006/12/25", "2007/01/01", "2007/01/15", "2007/02/19", "2007/05/28",
	"2007/07/04", "2007/09/03", "2007/10/08", "2007/11/12", "2007/11/22", "2007/12/25",
	"2008/01/01", "2008/01/21", "2008/02/18", "2008/05/26", "2008/07/04", "2008/09/01",
	"2008/10/13", "2008/11/11", "2008/11/27", "2008/12/25", "2009/01/01", "2009/01/19",
	"2009/02/16", "2009/05/25", "2009/07/03", "2009/09/07", "2009/10/12", "2009/11/11",
	"2009/11/26", "2009/12/25", "2010/01/01", "2010/01/18", "2010/02/15", "2010/05/31",
	"2010/07/05", "2010/09/06", "2010/10/11", "2010/11/11", "2010/11/25", "2010/12/24",
	"2010/12/31", "2011/01/17", "2011/02/21", "2011/05/30", "2011/07/04", "2011/09/05",
	"2011/10/10", "2011/11/11", "2011/11/24", "2011/12/26", "2012/01/02", "2012/01/16",
	"2012/02/20", "2012/05/28", "2012/07/04", "2012/09/03", "2012/10/08", "2012/11/12",
	"2012/11/22", "2012/12/25", "2013/01/01", "2013/01/21", "2013/02/18", "2013/05/27",
	"2013/07/04", "2013/09/02", "2013/10/14", "2013/11/11", "2013/11/28", "2013/12/25",
	"2014/01/01", "2014/01/20", "2014/02/17", "2014/05/26", "2014/07/04", "2014/09/01",
	"2014/10/13", "2014/11/11", "2014/11/27", "2014/12/25", "2015/01/01", "2015/01/19",
	"2015/02/16", "2015/05/25", "2015/07/03", "2015/09/07", "2015/10/12", "2015/11/11",
	"2015/11/26", "2015/12/25", "2016/01/01", "2016/01/18", "2016/02/15", "2016/05/30",
	"2016/07/04", "2016/09/05", "2016/10/10", "2016/11/11", "2016/11/24", "2016/12/26",
	"2017/01/02", "2017/01/16", "2017/02/20", "2017/05/29", "2017/07/04", "2017/09/04",
	"2017/10/09", "2017/11/10", "2017/11/23", "2017/12/25", "2018/01/01", "2018/01/15",
	"2018/02/19", "2018/05/28", "2018/07/04", "2018/09/03", "2018/10/08", "2018/11/12",
	"2018/11/22", "2018/12/25", "2019/01/01", "2019/01/21", "2019/02/18", "2019/05/27",
	"2019/07/04", "2019/09/02", "2019/10/14", "2019/11/11", "2019/11/28", "2019/12/25",
	"2020/01/01", "2020/01/20", "2020/02/17", "2020/05/25", "2020/07/03", "2020/09/07",
	"2020/10/12", "2020/11/11", "2020/11/26", "2020/12/25")))
}
