#' @export
#' @title Market-Clearing-Price Policy Function
#' @aliases policyMarketClearingPrice
#' @description This policy is to make the market clear every period.
#' In this case, the path of the economy is the spot market clearing path (alias instantaneous equilibrium path).
#' Generally, this function is passed to the function sdm2 as an argument to compute the spot market clearing path.
#' @param time the current time.
#' @param A a demand structure tree list (i.e. dstl, see demand_coefficient), a demand coefficient n-by-m matrix (alias demand structure matrix) or a function A(state) which returns an n-by-m matrix.
#' @param state the current state.
#' @param ... optional arguments to be passed to the function sdm2.
#' @return A list consisting of p, S and B which specify the prices, supplies and supply coefficient matrix after adjustment.
#' @references LI Wu (2019, ISBN: 9787521804225) General Equilibrium and Structural Dynamics: Perspectives of New Structural Economics. Beijing: Economic Science Press. (In Chinese)
#' @references Grandmont, J.M. (1977). Temporary General Equilibrium Theory. Econometrica 45, 535-572.
#' @seealso CGE::iep and \code{\link{sdm2}}, \code{\link{gemTemporaryEquilibriumPath}}.
#' The market clearing prices are the prices with a stickiness value equal to zero.
#' Therefore, this function can actually be replaced by \code{\link{makePolicyStickyPrice}} in the calculation.
#' @examples
#' \donttest{
#' #### an iep of the example (see Table 2.1 and 2.2) of the canonical dynamic
#' #### macroeconomic general equilibrium model in Torres (2016).
#' ge <- gemCanonicalDynamicMacroeconomic_3_2(
#'   policy.price = policyMarketClearingPrice,
#'   ts = TRUE,
#'   maxIteration = 1,
#'   numberOfPeriods = 50,
#'   z0 = c(0.5, 1)
#' )
#'
#' par(mfrow = c(1, 2))
#' matplot(ge$ts.z, type = "o", pch = 20)
#' matplot(ge$ts.p, type = "o", pch = 20)
#'
#' #### the same as above
#' ge <- gemCanonicalDynamicMacroeconomic_3_2(
#'   policy.price = makePolicyStickyPrice(stickiness = 0),
#'   ts = TRUE,
#'   maxIteration = 1,
#'   numberOfPeriods = 50,
#'   z0 = c(0.5, 1)
#' )
#'
#' par(mfrow = c(1, 2))
#' matplot(ge$ts.z, type = "o", pch = 20)
#' matplot(ge$ts.p, type = "o", pch = 20)
#'
#' #### TFP shock in the economy above (see Torres, 2016, section 2.8).
#' numberOfPeriods <- 200
#'
#' discount.factor <- 0.97
#' depreciation.rate <- 0.06
#' beta1.firm <- 0.35
#' return.rate <- 1 / discount.factor - 1
#'
#' set.seed(1)
#' alpha.shock <- rep(1, 100)
#' alpha.shock[101] <- exp(0.01)
#' for (t in 102:numberOfPeriods) {
#'   alpha.shock[t] <- exp(0.95 * log(alpha.shock[t - 1]))
#' }
#'
#' policyTechnologyChange <- function(time, A) {
#'   A[[1]]$func <- function(p) {
#'     result <- CD_A(
#'       alpha.shock[time], rbind(beta1.firm, 1 - beta1.firm, 0),
#'       c(p[1] * (return.rate + depreciation.rate), p[2:3])
#'     )
#'     result[3] <- p[1] * result[1] * return.rate / p[3]
#'     result
#'   }
#' }
#'
#' InitialEndowments <- {
#'   tmp <- matrix(0, 3, 2)
#'   tmp[1, 1] <- tmp[2, 2] <- tmp[3, 2] <- 1
#'   tmp
#' }
#'
#' ge <- gemCanonicalDynamicMacroeconomic_3_2(
#'   policy.supply = makePolicySupply(InitialEndowments),
#'   policy.technology = policyTechnologyChange,
#'   policy.price = policyMarketClearingPrice,
#'   ts = TRUE,
#'   maxIteration = 1,
#'   numberOfPeriods = 200
#' )
#'
#' c <- ge$A[1, 2] * ge$ts.z[, 2] # consumption
#' par(mfrow = c(2, 2))
#' matplot(ge$ts.z, type = "l")
#' x <- 100:140
#' plot(x, ge$ts.z[x, 1] / ge$ts.z[x[1], 1], type = "o", pch = 20)
#' plot(x, ge$ts.z[x, 2] / ge$ts.z[x[1], 2], type = "o", pch = 20)
#' plot(x, c[x] / c[x[1]], type = "o", pch = 20)
#'
#' #### an iep of example 7.2 (a monetary economy) in Li (2019). See CGE::Example7.2.
#' interest.rate <- 0.25
#' dst.firm <- node_new("cc", #composite commodity
#'                      type = "FIN",
#'                      rate = c(1, interest.rate),
#'                      "cc1", "money"
#' )
#' node_set(dst.firm, "cc1",
#'          type = "CD", alpha = 1, beta = c(0.5, 0.5),
#'          "wheat", "labor"
#' )
#'
#' dst.laborer <- Clone(dst.firm)
#' dst.money.lender <- Clone(dst.firm)
#'
#' dstl <- list(dst.firm, dst.laborer, dst.money.lender)
#'
#' B <- matrix(0, 3, 3)
#' B[1, 1] <- 1
#'
#' S0Exg <- matrix(NA, 3, 3)
#' S0Exg[2, 2] <- 100
#' S0Exg[3, 3] <- 100
#'
#' InitialEndowments <- {
#'   tmp <- matrix(0, 3, 3)
#'   tmp[1, 1] <- 10
#'   tmp[2, 2] <- tmp[3, 3] <- 100
#'   tmp
#' }
#'
#' ge <- sdm2(
#'   A = dstl, B = B, S0Exg = S0Exg,
#'   names.commodity = c("wheat", "labor", "money"),
#'   names.agent = c("firm", "laborer", "money.lender"),
#'   numeraire = c(money = interest.rate),
#'   numberOfPeriods = 20,
#'   maxIteration = 1,
#'   ts = TRUE,
#'   policy = list(
#'     makePolicySupply(S = InitialEndowments),
#'     policyMarketClearingPrice
#'   )
#' )
#'
#' par(mfrow = c(1, 2))
#' matplot(ge$ts.z, type = "o", pch = 20)
#' matplot(ge$ts.p, type = "o", pch = 20)
#' }

policyMarketClearingPrice <- function(time, A, state, ...) {
  instantaneous.equilibrium <- sdm2(A,
                                    B = 0 * state$S, S0Exg = state$S,
                                    names.commodity = state$names.commodity,
                                    names.agent = state$names.agent,
                                    ...
  )
  state$p <- instantaneous.equilibrium$p
  state
}


