% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/m_regression2.R
\name{regkienerLX}
\alias{regkienerLX}
\title{A Regression Function for Kiener Distributions}
\usage{
regkienerLX(X, model = "k4", dgts = c(3, 3, 1, 1, 1, 3, 2, 4, 4, 2, 2),
  maxk = 10, mink = 0.2, app = 0)
}
\arguments{
\item{X}{vector of quantiles.}

\item{model}{the model used for the regression: "k1", "k2", "k3", "k4".}

\item{dgts}{vector of length 11. Control the rounding of output parameters.}

\item{maxk}{numeric. The maximum value of tail parameter \code{k}.}

\item{mink}{numeric. The minimum value of tail parameter \code{k}.}

\item{app}{numeric. The parameter "\code{a}" in the function
                           \code{ppoints}.}
}
\value{
\item{dfrXP}{data.frame. X = initial quantiles. P = empirical probabilities.}
\item{dfrXL}{data.frame. X = initial quantiles. L = logit of probabilities.}
\item{dfrXR}{data.frame. X = initial quantiles. R = residuals after regression.}
\item{dfrEP}{data.frame. E = estimated quantiles. P = probabilities.}
\item{dfrEL}{data.frame. E = estimated quantiles. L = logit of probabilities.}
\item{dfrED}{data.frame. E = estimated quantiles.
                              D = estimated density (from probabilities).}
\item{regk0 }{object of class \code{nls} extracted from
              the regression function \code{\link[minpack.lm]{nlsLM}}.}
\item{coefk0}{the regression parameters in plain format.
              The median is out of the regression.}
\item{vcovk0}{rounded variance-covariance matrix.}
\item{vcovk0m}{rounded 1e+6 times variance-covariance matrix.}
\item{mcork0}{rounded correlation matrix.}
\item{coefk }{all parameters in plain format.}
\item{coefk1}{parameters for model "k1".}
\item{coefk2}{parameters for model "k2".}
\item{coefk3}{parameters for model "k3".}
\item{coefk4}{parameters for model "k4".}
\item{quantk}{quantiles of interest.}
\item{coefr }{all parameters in a rounded format.}
\item{coefr1}{rounded parameters for model "k1".}
\item{coefr2}{rounded parameters for model "k2".}
\item{coefr3}{rounded parameters for model "k3".}
\item{coefr4}{rounded parameters for model "k4".}
\item{quantr}{quantiles of interest in a rounded format.}
\item{dfrQkPk}{data.frame. Qk = Estimated quantiles of interest.
               Pk = probabilities.}
\item{dfrQkLk}{data.frame. Qk = Estimated quantiles of interest.
               Lk = Logit of probabilities.}
}
\description{
A function to estimate the distribution parameters of a given dataset with
Kiener distributions of type I, II, III and IV. It performs a nonlinear
regression of the logit of the empirical probabilities logit(p) on
quantiles X.
}
\details{
This function is designed to estimate the parameters of Kiener distributions
for a given dataset. It encapsulates the four distributions described in
this package.
"k1" uses model \code{lqkiener1}, "k2" uses model \code{lqkiener2},
"k3" uses model \code{lqkiener3} and "k4" uses model \code{lqkiener4}.

A typical input is a numeric vector that describes the returns of a stock.
Conversion from a (possible) time series format to a sorted numeric vector
is done automatically and without any check of the initial format.
There is also no check of missing values, \code{Na}, \code{NaN},
\code{-Inf}, \code{+Inf}.
Empirical probabilities of each point in the sorted dataset is calculated
with the function \code{\link[stats]{ppoints}}. The parameter \code{app}
corresponds to the parameter \code{a} in \code{ppoints} but has been
limited to the range (0, 0.5). Default value is 0 as large datasets are
very common in finance.

A nonlinear regression is performed with \code{\link[minpack.lm]{nlsLM}}
from the logit of the probabilities \code{logit(p)} over the quantiles X
with one of the functions \code{lqkiener1234}.
These functions have been selected as they
have an explicit form in the four types (this is unfortunately not the case
for \code{dkiener234}) and return satisfactory results with ordinary least
squares. The median is calculated before the regression and is injected
as a mandatory value in the regression function.

Kiener distributions use the following parameters, some of them being redundant.
See \code{\link{aw2k}} and \code{\link{pk2pk}} for the formulas and
the conversion between parameters:
\itemize{
  \item{ \code{m} (mu) is the median of the distribution,. }
  \item{ \code{g} (gamma) is the scale parameter. }
  \item{ \code{a} (alpha) is the left tail parameter. }
  \item{ \code{k} (kappa) is the harmonic mean of \code{a} and \code{w}
         and describes a global tail parameter. }
  \item{ \code{w} (omega) is the right tail parameter. }
  \item{ \code{d} (delta) is the distorsion parameter. }
  \item{ \code{e} (epsilon) is the eccentricity parameter. }
}
Where:
\itemize{
  \item{c(m, g, k) of length 3 for distribution "k1".}
  \item{c(m, g, a, w) of length 4 for distribution "k2".}
  \item{c(m, g, k, d) of length 4 for distribution "k3".}
  \item{c(m, g, k, e) of length 4 for distribution "k4".}
  \item{c(m, g, a, k, w, d, e) of length 7 extracted from object of class
        \code{clregk} like \code{regkienerLX} (typically \code{"reg$coefk"}).}
}

Model \code{"k1"} return results with 1+2=3 parameters and describes a
(assumed) symmetric distribution. Parameters \code{d} and \code{e} are set
to 0. Models \code{"k2"}, \code{"k3"} and \code{"k4"} describe asymmetric
distributions. They return results with 1+3=4 parameters.
Model "k2" has a very clear parameter definition but unfortunately
parameters \code{a} and \code{w} are highly correlated.
Model \code{"k3"} has the least correlated parameters but the meaning of
the distorsion parameter \code{d}, usually of order 1e-3, is not simple.
Multiplying it by 1000 might be a good choice but has not been done here.

Model \code{"k4"} exhibits a reasonable correlation between each parameter
and should be the preferred intermediate model between "k1" and "k2" models.
The eccentricity parameter \code{e} is well defined and easy to understand:
\eqn{e=(a-w)/(a+w)}, \eqn{a=k/(1-e)} and \eqn{w=k/(1+e)}. It is a relative
measure that could be expressed (and understood) as a percentage. Here, the
order 1e-2 has been preserved.

Tail parameter lower and upper values are controlled by \code{maxk} and
\code{mink}. An upper value \eqn{maxk = 10} is appropriate for datasets
of low and medium size, less than 50.000 points. For larger datasets, the
upper limit can be extended up to \eqn{maxk = 20}. Such a limit returns
results which are very closed to the logistic distribution, an alternate
distribution which could be more appropriate. The lower limit \code{mink}
is intended to avoid the value \eqn{k=0}. Remind
that value \eqn{k < 2} describes distribution with no stable variance and
\eqn{k < 1} describes distribution with no stable mean.

The output is an object in a flat format of class \code{clregk}. It can be
listed with the function \code{\link{attributes}}. First are the data.frames with
the initial data and the estimated results. Second is the result of
the regression \code{regk0} given by \code{\link[minpack.lm]{nlsLM}} from
which a few information have been extracted and listed here.
Third are the regression parameters (without the median) in plain format
(no rounding), the variance-covariance matrix, the variance-covariance
matrix times 1e+6 and the correlation matrix in a rounded format.
Note that \code{regk0}, \code{coefk0}, \code{coefk0tt}, \code{vcovk0},
\code{mcork0} have a polymorphic format and changing parameters that
depend from the selected model: "k1", "k2", "k3", "k4". They should be
used with care in subsequent calculations.
Fourth are the distribution parameters tailored to every model "k1", "k2",
"k3", "k4" plus estimated quantiles at levels:
c(0.001, 0.005, 0.01, 0.05, 0.5, 0.95, 0.99, 0.995, 0.999).
They are intended to subsequent calculations.
Fifth are the same parameters presented in a more readable format thanks
to the vector \code{dgts} which controls the rounding of the parameters in
the following order: \code{dgts = c("m", "g",
"a", "k", "w", "d", "e", "vcovk0", "vcovk0m", "mcork0", "quantr")}.
Sixth are the estimated quantiles and probabilities of interest stored
in a data.frame format.
}
\examples{
require(graphics)
require(minpack.lm)
require(timeSeries)

### Load the datasets and select one number (1-16)
DS     <- getDSdata()
j      <- 5


### and run this block
X      <- DS[[j]]
nameX  <- names(DS)[j]
reg    <- regkienerLX(X)

## Plotting
lleg   <- c("logit(0.999) = 6.9", "logit(0.99)   = 4.6",
           "logit(0.95)   = 2.9", "logit(0.50)   = 0",
           "logit(0.05)   = -2.9", "logit(0.01)   = -4.6",
           "logit(0.001) = -6.9  ")
pleg   <- c( paste("m =",  reg$coefr4[1]), paste("g  =", reg$coefr4[2]),
             paste("k  =", reg$coefr4[3]), paste("e  =", reg$coefr4[4]) )
op     <- par(mfrow=c(2,2), mgp=c(1.5,0.8,0), mar=c(3,3,2,1))
plot(X, type="l", main = nameX)
plot(reg$dfrXL, main = nameX, yaxt = "n")
axis(2, las=1, at=c(-9.2, -6.9, -4.6, -2.9, 0, 2.9, 4.6, 6.9, 9.2))
abline(h = c(-4.6, 4.6), lty = 4)
abline(v = c(reg$quantk[5], reg$quantk[9]), lty = 4)
legend("topleft", legend = lleg, cex = 0.7, inset = 0.02, bg = "#FFFFFF")
lines(reg$dfrEL, col = 2, lwd = 2)
points(reg$dfrQkLk, pch = 3, col = 2, lwd = 2, cex = 1.5)
plot(reg$dfrXP, main = nameX)
legend("topleft", legend = pleg, cex = 0.9, inset = 0.02 )
lines(reg$dfrEP, col = 2, lwd = 2)
plot(density(X), main = nameX)
lines(reg$dfrED, col = 2, lwd = 2)
round(cbind("k" = kmoments(reg$coefk, lengthx = nrow(reg$dfrXL)), "X" = xmoments(X)), 2)

## Attributes
attributes(reg)
head(reg$dfrXP)
head(reg$dfrXL)
head(reg$dfrXR)
head(reg$dfrEP)
head(reg$dfrEL)
head(reg$dfrED)
reg$regk0
reg$coefk0
reg$vcovk0
reg$vcovk0m
reg$mcork0
reg$coefk
reg$coefk1
reg$coefk2
reg$coefk3
reg$coefk4
reg$quantk
reg$coefr
reg$coefr1
reg$coefr2
reg$coefr3
reg$coefr4
reg$quantr
reg$dfrQkPk
reg$dfrQkLk
reg$fitk
### End block
}
\seealso{
\code{\link[minpack.lm]{nlsLM}}, \code{\link{laplacegaussnorm}},
    Kiener distributions of type I, II, III and IV: \code{\link{kiener1}}
    \code{\link{kiener2}}, \code{\link{kiener3}}, \code{\link{kiener4}}.
    Other functions: \code{\link{fitkienerLX}}, \code{\link{estimkienerX}}.
}

