# MIT License
# 
# Copyright (c) 2020 Nitesh Kumar, Abhinav Prakash, and Yu Ding
# 
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#   
#   The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

#' @title KNN : Predict
#' @description The function can be used to make prediction on test data using trained model
#' @param knnMdl a list containing:
#'  \itemize{
#'   \item knnMdl$data - The data set provided by user
#'   \item knnMdl$xCol - The column number of features provided by user or the best subset column number
#'   \item knnMdl$yCol - The column number of target provided by user
#'   \item knn$bestK - The best k nearest neighbor calculated using the function KnnFit
#'}
#'
#' @param testData a data frame or matrix, to compute the predictions
#' @examples 
#' 
#' data = data1[c(1:100),]
#' xCol = 2
#' yCol = 7
#' subsetSelection = FALSE
#' 
#' knn_model = KnnPCFit(data, xCol, yCol, subsetSelection)
#' testData = data1[c(101:110), ]
#' 
#' prediction = KnnPredict(knn_model, testData)
#' 
#' @return a numeric / vector with prediction on test data using model generated by KnnFit
#' @export
#'
KnnPredict = function(knnMdl, testData){

  trainData = knnMdl$data
  xCol = knnMdl$xCol
  yCol = knnMdl$yCol
  bestK = knnMdl$bestK
  
  normalizedTrainData = trainData
  normalizedTestData = testData
  
  for (feature in xCol) {
    
    normalizedTrainData[, feature] = (trainData[, feature] - min(trainData[, feature])) / (max(trainData[, feature]) - min(trainData[, feature]))
    normalizedTestData[, feature] = (testData[, feature] - min(trainData[, feature])) / (max(trainData[, feature]) - min(trainData[, feature]))
  }

  prediction = FNN::knn.reg(normalizedTrainData[, xCol, drop =FALSE], normalizedTestData[, xCol, drop = FALSE], trainData[, yCol], k = bestK )

  return(prediction$pred)
}
