% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/assist.R
\name{DAGassist}
\alias{DAGassist}
\title{Produce a compact \code{DAGassist} report (console/LaTeX/Word/Excel/Text)}
\usage{
DAGassist(
  dag,
  formula,
  data,
  exposure,
  outcome,
  engine = stats::lm,
  labels = NULL,
  verbose = TRUE,
  type = c("console", "latex", "word", "docx", "excel", "xlsx", "text", "txt"),
  out = NULL,
  imply = FALSE,
  omit_intercept = TRUE,
  omit_factors = TRUE,
  engine_args = list()
)
}
\arguments{
\item{dag}{A \strong{dagitty} object (see \code{\link[dagitty:dagitty]{dagitty::dagitty()}}).}

\item{formula}{Either (a) a standard model formula \code{Y ~ X + ...}, or
(b) a single \strong{engine call} such as \code{feols(Y ~ X + Z | fe, data = df, ...)}.
When an engine call is provided, \code{engine}, \code{data}, and extra arguments are
automatically extracted from the call.}

\item{data}{A \code{data.frame} (or compatible, e.g. tibble). Optional if supplied
via the engine call in \code{formula}.}

\item{exposure}{Optional character scalar; if missing/empty, inferred from the
DAG (must be unique).}

\item{outcome}{Optional character scalar; if missing/empty, inferred from the
DAG (must be unique).}

\item{engine}{Modeling function, default \link[stats:lm]{stats::lm}. Ignored if \code{formula}
is a single engine call (in that case the function is taken from the call).}

\item{labels}{Optional variable labels (named character vector or data.frame).}

\item{verbose}{Logical (default \code{TRUE}). Controls verbosity in the console
printer (formulas + notes).}

\item{type}{Output type. One of
\code{"console"} (default), \code{"latex"}/\code{"docx"}/\code{"word"},
\code{"excel"}/\code{"xlsx"}, \code{"text"}/\code{"txt"}.}

\item{out}{Output file path for the non-console types:
\itemize{
\item \code{type="latex"}: a \strong{LaTeX fragment} written to \code{out} (must end with \code{.tex}).
\item \code{type="docx"}/\code{"word"}: a \strong{Word (.docx)} file written to \code{out}.
\item \code{type="excel"}/\code{"xlsx"}: an \strong{Excel (.xlsx)} file written to \code{out}.
\item \code{type="text"}/\code{"txt"}: a \strong{plain-text} file written to \code{out}.
Ignored for \code{type="console"}.
}}

\item{imply}{Logical; default \code{FALSE}. \strong{Evaluation scope.}
\itemize{
\item If \code{FALSE} (default): restrict DAG evaluation to variables \strong{named in the formula}
(prune the DAG to exposure, outcome, and RHS terms). Roles/sets/bad-controls are
computed on this pruned graph, and the roles table \strong{only} shows those variables.
This is most useful if you want to refine your specific call.
\item If \code{TRUE}: evaluate on the \strong{full DAG} and allow DAG-implied controls in the
minimal/canonical sets; roles table shows all nodes. This is most useful if you
want to refine your overall control variable selection.
}}

\item{omit_intercept}{Logical; drop intercept rows from the model comparison (default \code{TRUE}).}

\item{omit_factors}{Logical; drop factor-level rows from the model comparison (default \code{TRUE}).}

\item{engine_args}{Named list of extra arguments forwarded to \code{engine(...)}.
If \code{formula} is an engine call, arguments from the call are merged with
\code{engine_args} (call values take precedence).}
}
\value{
An object of class \code{"DAGassist_report"}, invisibly for file outputs,
and printed for \code{type="console"}. The list contains:
\itemize{
\item \code{validation} - result from \code{validate_spec(...)} which verifies acyclicity and X/Y declarations.
\item \code{roles} - raw roles data.frame from \code{classify_nodes(...)} (logic columns).
\item \code{roles_display} - roles grid after labeling/renaming for exporters.
\item \code{bad_in_user} - variables in the user's RHS that are \code{MED}/\code{COL}/\code{IO}/\code{DMed}/\code{DCol}.
\item \code{controls_minimal} - (legacy) one minimal set (character vector).
\item \code{controls_minimal_all} - list of all minimal sets (character vectors).
\item \code{controls_canonical} - canonical set (character vector; may be empty).
\item \code{formulas} - list with \code{original}, \code{minimal}, \code{minimal_list}, \code{canonical}.
\item \code{models} - list with fitted models \code{original}, \code{minimal}, \code{minimal_list}, \code{canonical}.
\item \code{verbose}, \code{imply} - flags as provided.
}
}
\description{
\code{DAGassist()} validates a DAG + model specification, classifies node roles,
builds minimal and canonical adjustment sets, fits comparable models, and
renders a compact report in several formats (console, LaTeX fragment, DOCX,
XLSX, plain text). It also supports passing a \strong{single engine call} (e.g.
\code{feols(Y ~ X + Z | fe, data = df)}) instead of a plain formula.
}
\details{
\strong{Engine-call parsing.} If \code{formula} is a call (e.g., \code{feols(Y ~ X | fe, data=df)}),
DAGassist extracts the engine function, formula, data argument, and any additional
engine arguments directly from that call; these are merged with \code{engine}/\code{engine_args}
you pass explicitly (call arguments win).

\strong{Fixest tails.} For engines like \strong{fixest} that use \code{|} to denote FE/IV parts,
DAGassist preserves any \verb{| ...} tail when constructing minimal/canonical formulas
(e.g., \code{Y ~ X + controls | fe | iv(...)}).

\strong{Roles grid.} The roles table displays short headers:
\itemize{
\item \code{X} (exposure), \code{Y} (outcome), \code{CON} (confounder), \code{MED} (mediator),
\code{COL} (collider), \code{IO} (intermediate outcome = proper descendant of \code{Y}),
\code{DMed} (proper descendant of any mediator), \code{DCol} (proper descendant of any collider).
Descendants are \strong{proper} (exclude the node itself) and can be any distance downstream.
The internal \code{is_descendant_of_exposure} is retained for logic but hidden in displays.
}

\strong{Bad controls.} For total-effect estimation, DAGassist flags as \verb{bad controls}
any variables that are \code{MED}, \code{COL}, \code{IO}, \code{DMed}, or \code{DCol}. These are warned in
the console and omitted from the model-comparison table. Valid confounders (pre-treatment)
are eligible for minimal/canonical adjustment sets.

\strong{Output types.}
\itemize{
\item \code{console} prints roles, sets, formulas (if \code{verbose}), and a compact model comparison
with \code{{modelsummary}} if available (falls back gracefully otherwise).
\item \code{latex} writes a \strong{LaTeX fragment} you can \verb{\\\\input\{\}} into a paper.
\item \code{docx}/\code{word} writes a \strong{Word} doc (uses \code{options(DAGassist.ref_docx=...)} if set).
\item \code{excel}/\code{xlsx} writes an \strong{Excel} workbook with tidy tables.
\item \code{text}/\code{txt} writes a \strong{plain-text} report for logs/notes.
}

\strong{Dependencies.} Core requires \code{{dagitty}}. Optional enhancements: \code{{modelsummary}}
(pretty tables), \code{{broom}} (fallback tidying), \code{{rmarkdown}} + \strong{Pandoc} (DOCX),
\code{{writexl}} (XLSX).
}
\section{Interpreting the output}{

\strong{ROLES.} Variables in your formula are classified by DAG-based causal role:
\itemize{
\item \code{X} - treatment / exposure.
\item \code{Y} - outcome / dependent variable.
\item \code{CON} - confounder (common cause of \code{X} and \code{Y}); adjust for these.
\item \code{MED} - mediator (on a path from \code{X} to \code{Y}); do \strong{not} adjust when estimating total effects.
\item \code{COL} - collider (direct descendant of \code{X} and \code{Y}); adjusting opens a spurious path, so do \strong{not} adjust.
\item \code{IO} - intermediate outcome (descendant of \code{Y}); do \strong{not} adjust.
\item \code{DMed} - descendant of a mediator; do \strong{not} adjust when estimating total effects.
\item \code{DCol} - descendant of a collider; adjusting opens a spurious path, so do \strong{not} adjust.
\item \code{other} - safe, non-confounding predictors (e.g., affect \code{Y} only). Included in the canonical
model but omitted from the minimal set because they're not required for identification.
}

\strong{MODEL COMPARISON.}
\itemize{
\item \strong{Minimal} - the smallest adjustment set that blocks all back-door paths
(confounders only).
\item \strong{Canonical} - the largest permissible set: includes all controls that are not
\code{MED}, \code{COL}, \code{IO}, \code{DMed}, or \code{DCol}. \code{other} variables may appear here.
}
}

\section{Errors and edge cases}{

\itemize{
\item If exposure/outcome cannot be inferred uniquely, the function stops with a clear message.
\item Fitting errors (e.g., FE collinearity) are captured and displayed in comparisons
without aborting the whole pipeline.
}
}

\examples{
\dontshow{if (requireNamespace("dagitty", quietly = TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
\dontshow{set.seed(1)}
\dontshow{
# Build the DAG directly with dagitty
g <- dagitty::dagitty("dag { Z -> X; X -> M; X -> Y; M -> Y; Z -> Y; A -> Y; B -> Y; X -> C; Y -> C }")
dagitty::exposures(g) <- "X"; dagitty::outcomes(g) <- "Y"

n <- 150
A <- rnorm(n); B <- rnorm(n); Z <- rnorm(n)
X <- 0.8*Z + rnorm(n)
M <- 0.9*X + rnorm(n)
Y <- 0.7*X + 0.6*M + 0.3*Z + 0.2*A - 0.1*B + rnorm(n)
C <- 0.5*X + 0.4*Y + rnorm(n)
df <- data.frame(A,B,Z,X,M,Y,C)
}
# generate a console DAGassist report
DAGassist(dag = g, formula = lm(Y ~ X + Z + C + M, data = df))

# generate a LaTeX DAGassist report
\donttest{
DAGassist(dag = g, formula = lm(Y ~ X + Z + C + M, data = df),
          type = "latex", out = file.path(tempdir(), "frag.tex"))
}
\dontshow{\}) # examplesIf}
}
\seealso{
\code{\link[=print.DAGassist_report]{print.DAGassist_report()}} for the console printer, and the helper
exporters in \verb{report_*} modules.
}
