\name{SimulateCRF}
\alias{SimulateCRF}
\alias{Circular Random Field}
\title{Compute Circular Random Field Sample}

\description{
Generate an isotropic circular random field (CRF) in a plane.  A CRF is a space of circular random variables (CRV) with spatial dependence only dependent on distance.  CRFs are implemented for the circular distributions uniform (U), triangular (Tri), cardioid (Card), von Mises (vM), or wrapped Cauchy (WrC), with mean resultant direction (mu) of zero, and specified mean resultant length (rho) and range of spatial dependence
}

\usage{
SimulateCRF(N=100, CircDistr, Rho, Mu=0, Range, Ext=1, CovModel,
  Grid=NULL, Anisotropy=NULL, OverFit=FALSE, Resolution=.01)
}

\arguments{
  \item{N}{The number of observations to simulate}
  \item{CircDistr}{Name of circular distribution \dQuote{Card}, \dQuote{Tri}, \dQuote{U}, \dQuote{vM}, or \dQuote{WrC}}
  \item{Rho}{Mean resultant length parameter of circular distribution, see Details}
  \item{Mu}{Mean resultant direction of circular distribution (rad). abs(Mu) <= pi}
  \item{Range}{Distance at which CRV are not correlated}
  \item{Ext}{\var{Ext} >= 1.  \var{Range} x \var{Ext} (Extension) is the horizontal and vertical width of square sample space.}
  \item{CovModel}{Name of a spatial correlation function from \link[geoR]{cov.spatial}, e.g., \dQuote{exponential}}
  \item{Grid}{An n x 2 matrix of regular or irregular location coordinates of the simulated data.  \var{Grid} overides \var{N} and \var{Ext}.}
  \item{Anisotropy}{Vector of geometric anisotropy angle in radians, ratio > 1}.  See R Help for package geoR function grf.
  \item{OverFit}{If TRUE, standardize the GRV prior to transformation to CRV.  See Details.}
  \item{Resolution}{For trigonometric CDF, circular quantiles are interpolated at resolution Resolution.  0.001 <= Resolution <= 0.01   recommended.}
}

\details{
In the installed folder doc, Chapter5.CircularRandomFieldSimulation and Section J.2 in Appendices.J.PackageDocumentation provide additional detail and illustrations.\cr\cr

Circular-spatial dependence means that as the distance between measurement locations decreases, random rotations of direction from the mean direction tend to be more similar.  The CRF is obtained by transforming a gaussian random field (GRF),which is generated by \pkg{geoR} or \pkg{Random Fields} (if \var{N} > 500), via the inverse CDF of the CRV operating on the CDF of the gaussian random variable (GRV), which in turn, operates on observations of the GRV in the GRF.  For the uniform and triangular circular distributions, the inverse CDF is exact.  For the other trigonometric CDFs, the inverse CDF is estimated by interpolating a computed table of the CDF.  Because spatially dependent random variables (RV) tend to be similar at short distances and the CDF is monotonic increasing, their cumulative probabilities also tend to be similar.  Thus, spatial dependence is transfered from the GRF to the normal cumulative probabilities.  In turn, similarity of cumulative probabilities transfers to similarity of CRV via the monotonic inverse CDF of the CRV.\cr\cr

Details of the GRF are available in the geoR and RandomFields Help files.\cr\cr

OverFit=TRUE, realizations of the GRV are standardized (centered to mean 0 and scaled to standard deviation 1) prior to the transformation to a CRF.  Standardization stabilizes realizations of the GRV, enhancing the fit of the output CRF to the specified circular distribution.  Standardization is suitable for demonstration with closer fit, visualization, and illustrations.  Undesirable effects include loss of independence of the marginal GRVs, biased GRF covariance, and biased testing.  If FALSE (default), the realizations are not standardized.  Non-standardization includes expected variation from transformation of variation in mean and standard deviation of the realization of the GRV of the GRF.  OverFit=FALSE is recommended for the purposes of simulation, analysis, and testing.\cr\cr

Simulation at extension \eqn{Ext*Range - Range} beyond the range reduces edge effect at the range resulting in a more accurate representation of the sill at distances near the Range.  Limits of parameter rho: Tri (0,\eqn{4/pi^2}], Card (0,0.5], vM (0,1) and WrC (0,1).  The uniform circ distribution has rho=0.
}

\value{List of
 \item{x}{Vector of x coordinates of simulated observations.}
 \item{y}{Vector of y coordinates of simulated observations.}
 \item{direction}{Vector of direction of simulated observations in radians.}
 \item{Z}{Vector of simulated observations of GRV}
}

\seealso{
  \code{\link{CosinePlots}}\cr
  \code{\link{KrigCRF}}
}

\author{Bill Morphet}

\examples{
require(CircSpatial)
## Compute isotropic vM CRF of 121 observations, Rho=sqrt(0.5) so sill
## about 0.5, from GRF (Range=4, spherical covariance) with OverFit=TRUE
## for demonstration.
x1<- 1:11; y1 <- 1:11; y1 <- rep(y1, 11); x1 <- rep(x1, each=11)
set.seed(666)
crf1<- SimulateCRF(CircDistr="vM", Rho=sqrt(0.5), Range=4, CovModel=
  "spherical", Grid=cbind(x1, y1)) 
plot(crf1$x, crf1$y, type="n", xlab="", ylab="", asp=1)
arrow.plot(a1=crf1$x, a2=crf1$y, u=cos(crf1$direction), v=
  sin(crf1$direction), arrow.ex=0.1, xpd=TRUE, true.angle=TRUE,
  length=.1)
 
## Compute Cardioid CRF of 200 observations, Rho=0.4 so sill about 0.16,
## from GRF(exponential covariance, Range=5)
crf2 <- SimulateCRF(N=200, CircDistr="Card", Rho=0.4, Range=5, Ext=3,
  CovModel="exponential")

## Compute uniform CRF of 100 observations, sill about 0,
## from GRF(gaussian covariance, Range=8)
crf3 <- SimulateCRF(CircDistr="U", Range=8, Ext=3, CovModel="gaussian")

## Compute triangular CRF of 100 observations, sill about 0.04,
## from GRF(spherical covariance, Range=8)
crf4 <- SimulateCRF(CircDistr="Tri", Rho=0.5*4/pi^2, Range=8, Ext=3,
  CovModel="spherical")

## Compute wrapped Cauchy CRF of 100 observations, sill about 0.8,
## from GRF(exponential covariance, Range=8)
crf5 <- SimulateCRF(CircDistr="WrC", Rho=sqrt(0.8), Range=8, Ext=3,
  CovModel="exponential")

## Compute anisotropic wrapped Cauchy CRF of 400 observations, sill
## about 0.95, from GRF(spherical covariance, Range=8) with anisotropy
## angle pi/4 and ratio 3
crfaniso <- SimulateCRF(N=400, CircDistr="WrC", Rho=sqrt(0.95), Range=8,
  Ext=3, CovModel="spherical", Anisotropy=c(pi/4, 3))
}

\keyword{spatial}
\keyword{datagen}
