% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BuyseTest.R
\name{BuyseTest}
\alias{BuyseTest}
\title{Generalized Pairwise Comparisons (GPC)}
\usage{
BuyseTest(
  formula,
  data,
  scoring.rule = NULL,
  correction.uninf = NULL,
  model.tte = NULL,
  method.inference = NULL,
  n.resampling = NULL,
  strata.resampling = NULL,
  hierarchical = NULL,
  weight = NULL,
  neutral.as.uninf = NULL,
  keep.pairScore = NULL,
  seed = NULL,
  cpus = NULL,
  trace = NULL,
  treatment = NULL,
  endpoint = NULL,
  type = NULL,
  threshold = NULL,
  status = NULL,
  operator = NULL,
  censoring = NULL,
  strata = NULL,
  keep.comparison,
  method.tte
)
}
\arguments{
\item{formula}{[formula] a symbolic description of the GPC model,
typically \code{treatment ~ type1(endpoint1) + type2(endpoint2, threshold2) + strata}.
See Details, section "Specification of the GPC model".}

\item{data}{[data.frame] dataset.}

\item{scoring.rule}{[character] method used to compare the observations of a pair in presence of right censoring (i.e. \code{"timeToEvent"} endpoints).
Can be \code{"Gehan"} or \code{"Peron"}.
See Details, section "Handling missing values".}

\item{correction.uninf}{[integer] should a correction be applied to remove the bias due to the presence of uninformative pairs?
0 indicates no correction, 1 impute the average score of the informative pairs, and 2 performs IPCW.
See Details, section "Handling missing values".}

\item{model.tte}{[list] optional survival models relative to each time to each time to event endpoint.
Models must \code{prodlim} objects and stratified on the treatment and strata variable. When used, the uncertainty from the estimates of these survival models is ignored.}

\item{method.inference}{[character] method used to compute confidence intervals and p-values.
Can be \code{"none"}, \code{"u-statistic"}, \code{"permutation"}, \code{"studentized permutation"}, \code{"bootstrap"}, \code{"studentized bootstrap"}.
See Details, section "Statistical inference".}

\item{n.resampling}{[integer] the number of permutations/samples used for computing the confidence intervals and the p.values. 
See Details, section "Statistical inference".}

\item{strata.resampling}{[character] the variable on which the permutation/sampling should be stratified. 
See Details, section "Statistical inference".}

\item{hierarchical}{[logical] should only the uninformative pairs be analyzed at the lower priority endpoints (hierarchical GPC)?
Otherwise all pairs will be compaired for all endpoint (full GPC).}

\item{weight}{[numeric vector] weights used to cumulating the pairwise scores over the endpoints.
Only used when \code{hierarchical=FALSE}. Disregarded if the argument \code{formula} is defined.}

\item{neutral.as.uninf}{[logical vector] should paired classified as neutral be re-analyzed using endpoints of lower priority (as it is done for uninformative pairs).
See Details, section "Handling missing values".}

\item{keep.pairScore}{[logical] should the result of each pairwise comparison be kept?}

\item{seed}{[integer, >0] the seed to consider when performing resampling.
If \code{NULL} no seed is set.}

\item{cpus}{[integer, >0] the number of CPU to use.
Only the permutation test can use parallel computation.
See Details, section "Statistical inference".}

\item{trace}{[integer] should the execution of the function be traced ? \code{0} remains silent
and \code{1}-\code{3} correspond to a more and more verbose output in the console.}

\item{treatment, endpoint, type, threshold, status, operator, censoring, strata}{Alternative to \code{formula} for describing the GPC model.
See Details, section "Specification of the GPC model".}

\item{keep.comparison}{Obsolete. Alias for 'keep.pairScore'.}

\item{method.tte}{Obsolete. Alias for 'scoring.rule'.}
}
\value{
An \R object of class \code{\linkS4class{S4BuyseTest}}.
}
\description{
Performs Generalized Pairwise Comparisons for binary, continuous and time-to-event endpoints.
}
\details{
\bold{Specification of the GPC model}: \cr
There are two way to specify the GPC model in \code{BuyseTest}.
A \emph{Formula interface} via the argument \code{formula} where the response variable should be a binary variable defining the treatment arms. 
The rest of the formula should indicate the endpoints by order of priority and the strata variables (if any).
A \emph{Vector interface} using  the following arguments \itemize{
  \item \code{treatment}: [character] name of the treatment variable identifying the control and the experimental group.
Must have only two levels (e.g. \code{0} and \code{1}).
  \item \code{endpoint}: [character vector] the name of the endpoint variable(s).
  \item \code{threshold}: [numeric vector] critical values used to compare the pairs (threshold of minimal important difference).
A pair will be classified as neutral if the difference in endpoint is strictly below this threshold.
There must be one threshold for each endpoint variable; it must be \code{NA} for binary endpoints and positive for continuous or time to event endpoints. 
  \item \code{status}: [character vector] the name of the binary variable(s) indicating whether the endpoint was observed or censored.
Must value \code{NA} when the endpoint is not a time to event.
  \item \code{operator}: [character vector] the sign defining a favorable endpoint.
\code{">0"} indicates that higher values are favorable while "<0" indicates the opposite.
  \item \code{type}: [character vector] indicates whether it is
a binary outcome  (\code{"b"}, \code{"bin"}, or \code{"binary"}),
a continuous outcome  (\code{"c"}, \code{"cont"}, or \code{"continuous"}),
or a time to event outcome  (\code{"t"}, \code{"tte"}, \code{"time"}, or \code{"timetoevent"})
  \item \code{censoring}: [character vector] is the endpoint subject to right or left censoring (\code{"left"} or \code{"right"}). The default is right-censoring.
  \item \code{strata}: [character vector] if not \code{NULL}, the GPC will be applied within each group of patient defined by the strata variable(s).
}
The formula interface can be more concise, especially when considering few outcomes, but may be more difficult to apprehend for new users.
Note that arguments \code{endpoint}, \code{threshold}, \code{status}, \code{operator},  \code{type}, and \code{censoring} must have the same length. \cr \cr \cr


\bold{GPC procedure} \cr
The GPC procedure form all pairs of observations, one belonging to the experimental group and the other to the control group, and class them in 4 categories: \itemize{
 \item \emph{Favorable pair}: the endpoint is better for the observation in the experimental group.
 \item \emph{Unfavorable pair}: the endpoint is better for the observation in the control group.
 \item \emph{Neutral pair}: the difference between the endpoints of the two observations is (in absolute value) below the threshold. When \code{threshold=0}, neutral pairs correspond to pairs with equal endpoint. Lower-priority outcomes (if any) are then used to classified the pair into favorable/unfavorable.
 \item \emph{Uninformative pair}: censoring/missingness prevents from classifying into favorable, unfavorable or neutral.
}
With complete data, pairs can be decidely classified as favorable/unfavorable/neutral.
In presence of missing values, the GPC procedure uses the scoring rule (argument \code{scoring.rule}) and the correction for uninformative pairs (argument \code{correction.uninf}) to classify the pairs.
The classification may not be 0,1, e.g. the probability that the pair is favorable/unfavorable/neutral with the Peron's scoring rule.
To export the classification of each pair set the argument code{keep.pairScore} to \code{TRUE} and call the function \code{getPairScore} on the result of the \code{BuyseTest} function. \cr \cr \cr


\bold{Handling missing values}
\itemize{
  \item \code{scoring.rule}: indicates how to handle right-censoring in time to event endpoints.
The Gehan's scoring rule (argument \code{scoring.rule="Gehan"}) only scores pairs that can be decidedly classified as favorable, unfavorable, or neutral
while the "Peron"'s scoring rule (argument \code{scoring.rule="Peron"}) uses the empirical survival curves of each group to also score the pairs that cannot be decidedly classified.
The Peron's scoring rule is the recommanded scoring rule but only handles right-censoring.
  \item \code{correction.uninf}: indicates how to handle missing values that could not be classified by the scoring rule. \code{0} treat them as uninformative:
if \code{neutral.as.uninf=FALSE}  - this is an equivalent to complete case analysis -
while for \code{neutral.as.uninf=TRUE} uninformative pairs are treated as neutral, i.e., analyzed at the following endpoint (if any).
However both will (in general) lead to biased estimates for the proportion of favorable, unfavorable, or neutral pairs.
Inverse probability of censoring weights (IPCW, \code{correction.uninf=2}) is only recommanded when the analysis is stopped after the first endpoint with uninformative pairs.
Imputing the average score of the informative pairs (\code{correction.uninf=1}) is the recommanded approach.
Note that both corrections will convert the whole proportion of uninformative pairs of a given endpoint into favorable, unfavorable, or neutral pairs. \cr \cr
}


\bold{Statistical inference} \cr
The argument \code{method.inference} defines how to approximate the distribution of the GPC estimators and so how standard errors, confidence intervals, and p-values are computed.
Available methods are:
\itemize{
  \item argument \code{method.inference="none"}: only the point estimate is computed which makes the execution of the \code{BuyseTest} faster than with the other methods.
  \item argument \code{method.inference="u-statistic"}: uses a Gaussian approximation to obtain the distribution of the GPC estimators.
The U-statistic theory indicates that this approximation is asymptotically exact.
The variance is computed using a H-projection of order 1 (default option), which is a consistent but downward biased estimator.
An unbiased estimator can be obtained using a H-projection of order 2 (only available for the uncorrected Gehan's scoring rule, see \code{BuyseTest.options}).
\bold{WARNING}: the current implementation of the H-projection is not valid when using corrections for uninformative pairs (\code{correction.uninf=1}, or \code{correction.uninf=2}).
  \item argument \code{method.inference="permutation"}: perform a permutation test, estimating in each sample the summary statistics (net benefit, win ratio).
  \item argument \code{method.inference="studentized permutation"}: perform a permutation test, estimating in each sample the summary statistics (net benefit, win ratio) and the variance-covariance matrix of the estimate.
  \item argument \code{method.inference="bootstrap"}: perform a non-parametric boostrap, estimating in each sample the summary statistics (net benefit, win ratio).
  \item argument \code{method.inference=" studentized bootstrap"}: perform a non-parametric boostrap, estimating in each sample the summary statistics (net benefit, win ratio) and the variance-covariance matrix of the estimator.
}
Additional arguments for permutation and bootstrap resampling:
\itemize{
   \item \code{strata.resampling} If \code{NA} or of length 0, the permutation/non-parametric boostrap will be performed by resampling in the whole sample.
Otherwise, the permutation/non-parametric boostrap will be performed separately for each level that the variable defined in \code{strata.resampling} take.
   \item \code{n.resampling} set the number of permutations/samples used.
A large number of permutations (e.g. \code{n.resampling=10000}) are needed to obtain accurate CI and p.value. See (Buyse et al., 2010) for more details.
   \item \code{cpus} indicates whether the resampling procedure can be splitted on several cpus to save time. Can be set to \code{"all"} to use all available cpus.
The detection of the number of cpus relies on the \code{detectCores} function from the \emph{parallel} package. \cr \cr
}


\bold{Default values} \cr
The default of the arguments
\code{scoring.rule}, \code{correction.uninf}, \code{method.inference}, \code{n.resampling},
\code{hierarchical}, \code{neutral.as.uninf}, \code{keep.pairScore}, \code{strata.resampling},
\code{cpus}, \code{trace} is read from \code{BuyseTest.options()}. \cr
Additional (hidden) arguments are \itemize{
 \item \code{alternative} [character] the alternative hypothesis. Must be one of "two.sided", "greater" or "less" (used by \code{confint}).
 \item \code{conf.level} [numeric] level for the confidence intervals (used by \code{confint}).
 \item \code{keep.survival} [logical] export the survival values used by the Peron's scoring rule.
 \item \code{order.Hprojection} [1 or 2] the order of the H-projection used to compute the variance when \code{method.inference="u-statistic"}. 
}
}
\examples{
library(data.table)

#### simulate some data ####
set.seed(10)
df.data <- simBuyseTest(1e2, n.strata = 2)

## display 
if(require(prodlim)){
   resKM_tempo <- prodlim(Hist(eventtime,status)~treatment, data = df.data)
   plot(resKM_tempo)
}

#### one time to event endpoint ####
BT <- BuyseTest(treatment ~ TTE(eventtime, status = status), data= df.data)

summary(BT) # net benefit
summary(BT, percentage = FALSE)  
summary(BT, statistic = "winRatio") # win Ratio

## permutation instead of asymptotics to compute the p-value
\dontrun{
    BT <- BuyseTest(treatment ~ TTE(eventtime, status = status), data=df.data,
                    method.inference = "permutation", n.resampling = 1e3)
}
\dontshow{
    BT <- BuyseTest(treatment ~ TTE(eventtime, status = status), data=df.data,
                    method.inference = "permutation", n.resampling = 1e1, trace = 0)
}
summary(BT, statistic = "netBenefit") ## default
summary(BT, statistic = "winRatio") 

## parallel permutation
\dontrun{
    BT <- BuyseTest(treatment ~ TTE(eventtime, status = status), data=df.data,
                    method.inference = "permutation", n.resampling = 1e3, cpus = 2)
    summary(BT)
}

## method Gehan is much faster but does not optimally handle censored observations
BT <- BuyseTest(treatment ~ TTE(eventtime, status = status), data=df.data,
                scoring.rule = "Gehan", trace = 0)
summary(BT)

#### one time to event endpoint: only differences in survival over 1 unit ####
BT <- BuyseTest(treatment ~ TTE(eventtime, threshold = 1, status = status), data=df.data)
summary(BT)

#### one time to event endpoint with a strata variable
BT <- BuyseTest(treatment ~ strata + TTE(eventtime, status = status), data=df.data)
summary(BT)

#### several endpoints with a strata variable
f <- treatment ~ strata + T(eventtime, status, 1) + B(toxicity) 
f <- update(f, 
            ~. + T(eventtime, status, 0.5) + C(score, 1) + T(eventtime, status, 0.25))

BT <- BuyseTest(f, data=df.data)
summary(BT)

#### real example : veteran dataset of the survival package ####
## Only one endpoint. Type = Time-to-event. Thresold = 0. Stratfication by histological subtype
## scoring.rule = "Gehan"

if(require(survival)){
\dontrun{
  library(survival) ## import veteran
 
  ## scoring.rule = "Gehan"
  BT_Gehan <- BuyseTest(trt ~ celltype + TTE(time,threshold=0,status=status), 
                        data=veteran, scoring.rule="Gehan")
  
  summary_Gehan <- summary(BT_Gehan)
  summary_Gehan <- summary(BT_Gehan, statistic = "winRatio")
  
  ## scoring.rule = "Peron"
  BT_Peron <- BuyseTest(trt ~ celltype + TTE(time,threshold=0,status=status), 
                        data=veteran, scoring.rule="Peron")

  class(BT_Peron)
  summary(BT_Peron)
}
}
}
\references{
On the GPC procedure: Marc Buyse (2010). \bold{Generalized pairwise comparisons of prioritized endpoints in the two-sample problem}. \emph{Statistics in Medicine} 29:3245-3257 \cr
On the win ratio: D. Wang, S. Pocock (2016). \bold{A win ratio approach to comparing continuous non-normal outcomes in clinical trials}. \emph{Pharmaceutical Statistics} 15:238-245 \cr
On the Peron's scoring rule: J. Peron, M. Buyse, B. Ozenne, L. Roche and P. Roy (2018). \bold{An extension of generalized pairwise comparisons for prioritized outcomes in the presence of censoring}. \emph{Statistical Methods in Medical Research} 27: 1230-1239  \cr 
On the Gehan's scoring rule: Gehan EA (1965). \bold{A generalized two-sample Wilcoxon test for doubly censored data}. \emph{Biometrika}  52(3):650-653 \cr
On inference in GPC using the U-statistic theory: I. Bebu, J. M. Lachin (2015). \bold{Large sample inference for a win ratio analysis of a composite outcome based on prioritized components}. \emph{Biostatistics} 17(1):178-187 \cr
}
\seealso{
\code{\link{S4BuyseTest-summary}} for a summary of the results of generalized pairwise comparison. \cr
\code{\link{S4BuyseTest-class}} for a presentation of the \code{S4BuyseTest} object. \cr
\code{\link{constStrata}} to create a strata variable from several clinical variables. \cr
}
\author{
Brice Ozenne
}
\keyword{BuyseTest}
\keyword{function}
