.plotgridsom = function(x, xlim, ylim, ...)
{
  ## The following two lines leave equal amounts of space on both
  ## sides of the plot if no xlim or ylim are given
  if (missing(xlim)) xlim <- c(0, max(x$pts[,1]) + min(x$pts[,1]))
  if (missing(ylim)) ylim <-  c(max(x$pts[,2]) + min(x$pts[,2]), 0)
  MASS::eqscplot(xlim, ylim, axes = FALSE,
           type = "n", xlab = "", ylab = "", ...)
}

#' Plot self-organising map
#' 
#' Plot self-organising map
#' 
#' Plot self-organising map, obtained from function kohonen. shows where objects 
#' are mapped. It needs the \code{classif} argument, and a \code{labels} or 
#' \code{pchs} argument.
#' 
#' @param x Kohonen object.
#' 
#' @param main Title of the plot.
#' 
#' @param index The location of the user's objective data
#'             
#' @param label Labels to plot.
#'              
#' @param pchs Symbols to plot.
#'               
#' @param bgcol Optional argument to colour the unit backgrounds for the "mapping" and 
#'              "codes" plotting type. Defaults to "gray" and "transparent" in both types, 
#'              respectively.
#'              
#' @param cols A specification for the default number color. 
#' 
#' @param lwds The number width, a positive number, defaulting to 1. 
#'                        
#' @param keepMargins If \code{FALSE} (the default), restore the original graphical 
#'                    parameters after plotting the kohonen map. If \code{TRUE}, one retains 
#'                    the map coordinate system so that one can add symbols to the plot, or 
#'                    map unit numbers using the \code{identify} function.
#' 
#' @param ... other graphical parameters. 
#' 
#' @return plot mapping                   
#'                                       
#' @keywords clusterPlotSOMmap 
#'
#' @aliases clusterPlotSOMmap
#' 
#' @author Min-feng Zhu <\email{wind2zhu@@163.com}>, 
#' 
#' @export clusterPlotSOMmap
#'  
#' @examples
#' data(som.bcl)
#' 
#' clusterPlotSOMmap(som.bcl, pchs = 1)
#' idx = unique(som.bcl$unit.classif[1:80])
#' bgcols <- c("gray", "lightgreen")
#' index = rep(1, 5 * 5)
#' index[idx]  = 2
#' bgcol = bgcols[as.integer(index)]
#' label = c()
#' for (i in 1:length(idx)) {
#'   label[i] = length(which(som.bcl$unit.classif[1:80] == idx[i]))
#'   }
#' 
#' clusterPlotSOMmap(som.bcl, index = idx, label = label, bgcol = bgcol)
#' 
clusterPlotSOMmap <- function(x, index = NULL, main = NULL, label = NULL, pchs = 46, bgcol = NULL, 
                                        cols = 'red', lwds = 2, keepMargins = FALSE, ...)
{
  if (is.null(main)) main <- "Mapping plot"
  
  margins <- rep(0.6, 4)
  if (main != "") margins[3] <- margins[3] + 2
  if (!keepMargins) {
    opar <- par("mar")
    on.exit(par(mar = opar))
  }
  par(mar=margins)
  
  if (!is.null(x$unit.classif)) {
    classif <- x$unit.classif
  } else if (is.null(classif))
    stop("No mapping available")
  .plotgridsom(x$grid, ...)
  
  title.y <- max(x$grid$pts[,2]) + 1.2
  if (title.y > par("usr")[4] - .2){
    title(main)
  } else {
    text(mean(range(x$grid$pts[,1])),
         title.y,
         main, adj = .5, cex = par("cex.main"),
         font = par("font.main"))
  }
  
  if (is.null(bgcol)) bgcol <- "gray"
  symbols(x$grid$pts[, 1], x$grid$pts[, 2],
          squares = rep(1, nrow(x$grid$pts)),
          inches = FALSE, add = TRUE, bg = bgcol)
  if (!is.null(pchs))
    points(x$grid$pts[classif, 1] + rnorm(length(classif), 0, 0.12),
           x$grid$pts[classif, 2] + rnorm(length(classif), 0, 0.12),
           pch = pchs, ...)
  if (!is.null(index) & !is.null(label))
    text(x$grid$pts[index, 1] + rnorm(length(index), 0, 0.12),
         x$grid$pts[index, 2] + rnorm(length(index), 0, 0.12),
         labels = label, lwd = lwds, col = cols, ...)
  
  
  invisible()
}
