% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/posterior_rt.R
\name{posterior_rt}
\alias{posterior_rt}
\title{Estimate the effective reproduction number with the next generation matrix approach}
\usage{
posterior_rt(object, y_data, age_distribution_population, infectious_period)
}
\arguments{
\item{object}{An object of class \code{stanigbm}. See \code{\link[Bernadette]{stan_igbm}}.}

\item{y_data}{data.frame;
age-specific mortality counts in time. See \code{data(age_specific_mortality_counts)}.}

\item{age_distribution_population}{data.frame;
the age distribution of a given population. See \code{aggregate_age_distribution}.}

\item{infectious_period}{integer;
length of infectious period in days. Must be >=1.}
}
\value{
A data.frame which can be visualised using \code{\link[Bernadette]{plot_posterior_rt}}.
}
\description{
Estimate the effective reproduction number with the next generation matrix approach
}
\examples{
\donttest{
# Age-specific mortality/incidence count time series:
data(age_specific_mortality_counts)
data(age_specific_infection_counts)

# Import the age distribution for Greece in 2020:
age_distr <- age_distribution(country = "Greece", year = 2020)

# Lookup table:
lookup_table <- data.frame(Initial = age_distr$AgeGrp,
                          Mapping = c(rep("0-39",  8),
                                      rep("40-64", 5),
                                      rep("65+"  , 3)))

# Aggregate the age distribution table:
aggr_age <- aggregate_age_distribution(age_distr, lookup_table)

# Import the projected contact matrix for Greece:
conmat <- contact_matrix(country = "GRC")

# Aggregate the contact matrix:
aggr_cm <- aggregate_contact_matrix(conmat, lookup_table, aggr_age)

# Aggregate the IFR:
ifr_mapping <- c(rep("0-39", 8), rep("40-64", 5), rep("65+", 3))

aggr_age_ifr <- aggregate_ifr_react(age_distr, ifr_mapping, age_specific_infection_counts)

# Infection-to-death distribution:
ditd <- itd_distribution(ts_length  = nrow(age_specific_mortality_counts),
                         gamma_mean = 24.19231,
                         gamma_cv   = 0.3987261)

# Posterior sampling:

rstan_options(auto_write = TRUE)
chains <- 2
options(mc.cores = chains)

igbm_fit <- stan_igbm(y_data                      = age_specific_mortality_counts,
                      contact_matrix              = aggr_cm,
                      age_distribution_population = aggr_age,
                      age_specific_ifr            = aggr_age_ifr[[3]],
                      itd_distr                   = ditd,
                      incubation_period           = 3,
                      infectious_period           = 4,
                      likelihood_variance_type    = "linear",
                      prior_scale_x0              = 0.5,
                      prior_scale_contactmatrix   = 0.05,
                      pi_perc                     = 0.1,
                      prior_volatility            = normal(location = 0, scale = 1),
                      prior_nb_dispersion         = exponential(rate = 1/5),
                      algorithm_inference         = "sampling",
                      nBurn                       = 5,
                      nPost                       = 10,
                      nThin                       = 1,
                      chains                      = chains,
                      adapt_delta                 = 0.8,
                      max_treedepth               = 16,
                      seed                        = 1)

post_rt_summary <- posterior_rt(object                      = igbm_fit,
                                y_data                      = age_specific_mortality_counts,
                                age_distribution_population = aggr_age,
                                infectious_period            = 4)

# Visualise the posterior distribution of the effective reproduction number:
plot_posterior_rt(post_rt_summary)
}
}
\references{
Diekmann, O., Heesterbeek, J., and Roberts, M. (2010). The construction of next-generation matrices for compartmental epidemic models. \emph{J. R. Soc. Interface}, 7, 873–-885.

Bouranis, L., Demiris, N. Kalogeropoulos, K. and Ntzoufras, I. (2022). Bayesian analysis of diffusion-driven multi-type epidemic models with application to COVID-19. arXiv: \url{https://arxiv.org/abs/2211.15229}
}
