% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get.oc.R
\name{get.oc}
\alias{get.oc}
\title{Generate operating characteristics for single agent trials}
\usage{
get.oc(target, p.true, ncohort, cohortsize, n.earlystop=100,
              startdose=1, titration=FALSE, p.saf=0.6*target, p.tox=1.4*target,
              cutoff.eli=0.95,extrasafe=FALSE, offset=0.05, boundMTD=FALSE,
              ntrial=1000, seed=6)
}
\arguments{
\item{target}{the target DLT rate}

\item{p.true}{a vector containing the true toxicity probabilities of the
investigational dose levels.}

\item{ncohort}{the total number of cohorts}

\item{cohortsize}{the cohort size}

\item{n.earlystop}{the early stopping parameter. If the number of patients
treated at the current dose reaches \code{n.earlystop},
stop the trial and select the MTD based on the observed data.
The default value \code{n.earlystop=100} essentially turns
off this type of early stopping.}

\item{startdose}{the starting dose level for the trial}

\item{titration}{set \code{titration=TRUE} to perform dose escalation with cohort size = 1 to accelerate dose escalation at the begining of the trial.}

\item{p.saf}{the highest toxicity probability that is deemed subtherapeutic
(i.e. below the MTD) such that dose escalation should be undertaken.
The default value is \code{p.saf=0.6*target}.}

\item{p.tox}{the lowest toxicity probability that is deemed overly toxic such
that deescalation is required. The default value is
\code{p.tox=1.4*target}).}

\item{cutoff.eli}{the cutoff to eliminate an overly toxic dose for safety.
We recommend the default value of (\code{cutoff.eli=0.95}) for general use.}

\item{extrasafe}{set \code{extrasafe=TRUE} to impose a more stringent stopping rule}

\item{offset}{a small positive number (between \code{0} and \code{0.5}) to control how strict the
stopping rule is when \code{extrasafe=TRUE}. A larger value leads to a more
strict stopping rule. The default value \code{offset=0.05} generally works well.}

\item{boundMTD}{set \code{boundMTD=TRUE} to impose the condition: the isotonic estimate of toxicity probability
for the selected MTD must be less than de-escalation boundary.}

\item{ntrial}{the total number of trials to be simulated}

\item{seed}{the random seed for simulation}
}
\value{
\code{get.oc()} returns the operating characteristics of the BOIN design as a list,
       including:
       (1) selection percentage at each dose level (\code{$selpercent}),
       (2) the number of patients treated at each dose level (\code{$npatients}),
       (3) the number of toxicities observed at each dose level (\code{$ntox}),
       (4) the average number of toxicities (\code{$totaltox}),
       (5) the average number of patients (\code{$totaln}),
       (6) the percentage of early stopping without selecting the MTD (\code{$percentstop}),
       (7) risk of overdosing 60\% or more of patients (\code{$overdose60}),
       (8) risk of overdosing 80\% or more of patients (\code{$overdose80}),
       (9) data.frame (\code{$simu.setup}) containing simulation parameters, such as target, p.true, etc.
}
\description{
Obtain the operating characteristics of the BOIN design for single agent trials by simulating trials.
}
\details{
The operating characteristics of the BOIN design are generated by simulating trials
         under the prespecified true toxicity probabilities of the investigational doses. If
         \code{titration=TRUE}, we perform dose escalation with cohort size = 1 at the begining of the trial:
         starting from \code{startdose}, if no toxicity is observed, we escalate the dose;
         otherwise, the titration is completed and we switch to cohort size = \code{cohortsize}.
         Titration accelerates the dose escalation and is useful when low doses are believed to be safe.


         The BOIN design has two built-in stopping rules: (1) stop the trial if the lowest
         dose is eliminated due to toxicity, and no dose should be selected as the MTD; and
         (2) stop the trial and select the MTD if the number of patients treated at the current
         dose reaches \code{n.earlystop}. The first stopping rule is a safety rule to protect patients
         from the case in which all doses are overly toxic. The rationale for the second stopping
         rule is that when there is a large number (i.e., \code{n.earlystop}) of patients
         assigned to a dose, it means that the dose-finding algorithm has approximately converged.
         Thus, we can stop the trial early and select the MTD to save sample size and reduce the
         trial duration. For some applications, investigators may prefer a more strict safety
         stopping rule than rule (1) for extra safety when the lowest dose is overly toxic.
         This can be achieved by setting \code{extrasafe=TRUE}, which imposes the following more
         strict safety stopping rule: stop the trial if (i) the number of patients treated at the
         lowest dose \code{>=3}, and (ii) \eqn{Pr(toxicity\ rate\ of\ the\ lowest\ dose > \code{target} | data)
         > \code{cutoff.eli}-\code{offset}}. As a tradeoff, the strong stopping rule will decrease the MTD
         selection percentage when the lowest dose actually is the MTD.
}
\note{
We should avoid setting the values of \code{p.saf} and \code{p.tox} very close to the
     \code{target}. This is because the small sample sizes of typical phase I trials prevent us from
      differentiating the target DLT rate from the rates close to it. The default values provided by
      \code{get.oc()} are strongly recommended, and generally yield excellent operating characteristics.
}
\examples{

## get the operating characteristics for BOIN single agent trial
oc <- get.oc(target=0.3, p.true=c(0.05, 0.15, 0.3, 0.45, 0.6),
			ncohort=20, cohortsize=3, ntrial=1000)

summary(oc) # summarize design operating characteristics
plot(oc)   # plot flowchart of the BOIN design and design operating characteristics, including
           # selection percentage, number of patients, and observed toxicities at each dose


## perform titration at the begining of the trial to accelerate dose escalation
oc <- get.oc(target=0.3, p.true=c(0.05, 0.15, 0.3, 0.45, 0.6),
			titration=TRUE, ncohort=20, cohortsize=3, ntrial=1000)

summary(oc)          # summarize design operating characteristics
plot(oc)  # plot flowchart of the BOIN design and design operating characteristics
}
\references{
Liu S. and Yuan, Y. (2015). Bayesian Optimal Interval Designs for Phase I
            Clinical Trials, \emph{Journal of the Royal Statistical Society: Series C}, 64, 507-523.

           Yan, F., Zhang, L., Zhou, Y., Pan, H., Liu, S. and Yuan, Y. (2020).BOIN: An R Package
           for Designing Single-Agent and Drug-Combination Dose-Finding Trials Using Bayesian Optimal
           Interval Designs. \emph{Journal of Statistical Software}, 94(13),1-32.<doi:10.18637/jss.v094.i13>.



            Yuan Y., Hess K.R., Hilsenbeck S.G. and Gilbert M.R. (2016) Bayesian Optimal Interval Design: A
       Simple and Well-performing Design for Phase I Oncology Trials, \emph{Clinical Cancer Research}, 22, 4291-4301.
}
\seealso{
Tutorial: \url{http://odin.mdacc.tmc.edu/~yyuan/Software/BOIN/BOIN2.6_tutorial.pdf}

         Paper: \url{http://odin.mdacc.tmc.edu/~yyuan/Software/BOIN/paper.pdf}
}
\author{
Suyu Liu, Yanhong Zhou, and Ying Yuan
}
