% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fevd.R
\name{fevd}
\alias{fevd}
\alias{fevd.bgvar.irf}
\title{Forecast Error Variance Decomposition}
\usage{
fevd(x, R=NULL, var.slct=NULL, verbose=TRUE)
}
\arguments{
\item{x}{an object of class \code{bgvar.irf}.}

\item{R}{If \code{NULL} and the \code{x} has been fitted via sign restrictions, the rotation matrix is used that minimizes the distance to the median impulse responses at the posterior median.}

\item{var.slct}{character vector that contains the variables for which forecast error variance decomposition should be performed. If \code{NULL} the FEVD is computed for the whole system, which is very time consuming.}

\item{verbose}{If set to \code{FALSE} it suppresses printing messages to the console.}
}
\value{
Returns a list with two elements \itemize{
\item{\code{FEVD}}{  an array of size (K times horizon times N), where K are all variables in the system, horizon is the specified impulse response horizon and N is the size of the decomposed structural variables (if \code{var.slct=NULL} then K=N).}
\item{\code{xglobal}}{ used data of the model.}
}
}
\description{
This function calculates the forecast error variance decomposition (FEVDs) for Cholesky and sign-identified shocks.
}
\details{
Since the calculations are very time consuming, the FEVDs are based on the posterior median only (as opposed to calculating FEVDs for each MCMC sweep). In case the underlying shock has been identified via sign restrictions, the rotation matrix corresponds to the one that fulfills the sign restrictions at the posterior median of the estimated coefficients. More precisely, the algorithm searches for 50 rotation matrices that fulfill the sign restrictions at the \emph{posterior median} of the coefficients and then singles out the rotation matrix that minimizes the distance to the median of the impulse responses as suggested in Fry and Pagan (2011).
}
\examples{
\dontshow{
library(BGVAR)
data(eerData)
cN<-c("EA","US","UK")
eerData<-eerData[cN]
W.trade0012<-apply(W.trade0012[cN,cN],2,function(x)x/rowSums(W.trade0012[cN,cN]))
model.ssvs.eer<-bgvar(Data=eerData,W=W.trade0012,draws=50,burnin=50,plag=1,
                      prior="SSVS",thin=1,eigen=TRUE)
                      
# US monetary policy shock
shocks<-list();shocks$var="stir";shocks$cN<-"US";shocks$ident="chol";shocks$scal=-100
irf.chol.us.mp<-irf(model.ssvs.eer,shock=shocks,n.ahead=48)

# calculates FEVD for variables US.Dp and EA.y
fevd.us.mp=fevd(irf.chol.us.mp,var.slct=c("US.Dp","EA.y"))

# US monetary policy shock with sign restrictions
sign.constr<-list()
sign.constr$shock1$shock             <- c("US.stir")
sign.constr$shock1$restrictions$res1 <- c("US.y")
sign.constr$shock1$restrictions$res2 <- c("US.Dp")
sign.constr$shock1$sign              <- c(">","<","<")
sign.constr$shock1$rest.horz         <- c(1,1,1)
sign.constr$shock1$constr            <- c(1,1,1)
sign.constr$shock1$scal              <- +100 
sign.constr$MaxTries<-200
irf.sign.us.mp<-irf(model.ssvs.eer,sign.constr=sign.constr,n.ahead=24)

# calculates FEVD for variables US.Dp and EA.y
fevd.us.mp=fevd(irf.sign.us.mp,var.slct=c("US.Dp","EA.y"))
}
\donttest{
library(BGVAR)
data(eerData)
model.ssvs.eer<-bgvar(Data=eerData,W=W.trade0012,draws=100,burnin=100,plag=1,
                      prior="SSVS",thin=1,eigen=TRUE)
                      
# US monetary policy shock
shocks<-list();shocks$var="stir";shocks$cN<-"US";shocks$ident="chol";shocks$scal=-100
irf.chol.us.mp<-irf(model.ssvs.eer,shock=shocks,n.ahead=48)

# calculates FEVD for variables US.Dp and EA.y
fevd.us.mp=fevd(irf.chol.us.mp,var.slct=c("US.Dp","EA.y"))

# US monetary policy shock with sign restrictions
sign.constr<-list()
sign.constr$shock1$shock             <- c("US.stir")
sign.constr$shock1$restrictions$res1 <- c("US.y")
sign.constr$shock1$restrictions$res2 <- c("US.Dp")
sign.constr$shock1$sign              <- c(">","<","<")
sign.constr$shock1$rest.horz         <- c(1,1,1)
sign.constr$shock1$constr            <- c(1,1,1)
sign.constr$shock1$scal              <- +100 
sign.constr$MaxTries<-200
irf.sign.us.mp<-irf(model.ssvs.eer,sign.constr=sign.constr,n.ahead=24)

# calculates FEVD for variables US.Dp and EA.y
fevd.us.mp=fevd(irf.sign.us.mp,var.slct=c("US.Dp","EA.y"))
}
# NOT RUN - calculates FEVDs for all variables in the system, very time consuming
\dontrun{
}
}
\seealso{
\code{\link{irf}}
}
\author{
Maximilian Boeck, Martin Feldkircher, Florian Huber
}
